% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtstm.R
\name{CohortDtstm}
\alias{CohortDtstm}
\title{Cohort discrete time state transition model}
\format{
An \link[R6:R6Class]{R6::R6Class} object.
}
\description{
Simulate outcomes from a cohort discrete time state transition model.
}
\examples{
library("data.table")
library("nnet")
transitions_data <- data.table(multinom3_exdata$transitions)

# Treatment strategies, target population, and model structure
n_patients <- 4
patients <- transitions_data[year == 1, .(patient_id, age, female)][
 sort(sample.int(nrow(transitions_data[year == 1]), n_patients))][
   , grp_id := 1:n_patients]
hesim_dat <- hesim_data(
  patients = patients,
  strategies = data.table(strategy_id = 1:2,
                          strategy_name = c("Reference", "Intervention")),
  states = data.table(state_id = c(1, 2),
                      state_name = c("Healthy", "Sick")) # Non-death health states
)
tmat <- rbind(c(0, 1, 2),
              c(NA, 0, 1),
              c(NA, NA, NA))

# Parameter estimation
## Multinomial logistic regression
data_healthy <- transitions_data[state_from == "Healthy"]
fit_healthy <- multinom(state_to ~ strategy_name + female + age,
                        data = data_healthy, trace = FALSE)
data_sick <- droplevels(transitions_data[state_from == "Sick"])
fit_sick <- multinom(state_to ~ strategy_name + female + age,
                     data = data_sick, trace = FALSE)
transfits <- multinom_list(healthy = fit_healthy, sick = fit_sick)

## Utility
utility_tbl <- stateval_tbl(multinom3_exdata$utility,
                            dist = "beta",
                            hesim_data = hesim_dat)

## Costs
drugcost_tbl <- stateval_tbl(multinom3_exdata$costs$drugs,
                             dist = "fixed",
                             hesim_data = hesim_dat) 
medcost_tbl <- stateval_tbl(multinom3_exdata$costs$medical,
                            dist = "gamma",
                            hesim_data = hesim_dat)  

# Economic model
n_samples <- 3

## Construct model
### Transitions
transmod_data <- expand(hesim_dat)
transmod <- create_CohortDtstmTrans(transfits,
                                    input_data = transmod_data,
                                    trans_mat = tmat,
                                    n = n_samples,
                                    point_estimate = FALSE)

### Utility
utilitymod <- create_StateVals(utility_tbl, n = n_samples)

### Costs
drugcostmod <- create_StateVals(drugcost_tbl, n = n_samples)
medcostmod <- create_StateVals(medcost_tbl, n = n_samples)
costmods <- list(Drug = drugcostmod,
                 Medical = medcostmod)

### Combine
econmod <- CohortDtstm$new(trans_model = transmod,
                           utility_model = utilitymod,
                           cost_models = costmods)

## Simulate outcomes
econmod$sim_stateprobs(n_cycles = 20)
econmod$sim_qalys(dr = .03)
econmod$sim_costs(dr = .03)
econmod$summarize()
econmod$summarize(by_grp = TRUE)

}
\seealso{
\code{\link[=create_CohortDtstm]{create_CohortDtstm()}}, \link{CohortDtstmTrans},
\code{\link[=create_CohortDtstmTrans]{create_CohortDtstmTrans()}}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{trans_model}}{The model for health state transitions. Must be an object
of class \link{CohortDtstmTrans}.}

\item{\code{utility_model}}{The model for health state utility. Must be an object of
class \link{StateVals}.}

\item{\code{cost_models}}{The models used to predict costs by health state.
Must be a list of objects of class \link{StateVals}, where each element of the
list represents a different cost category.}

\item{\code{stateprobs_}}{An object of class \link{stateprobs} simulated using \verb{$sim_stateprobs()}.}

\item{\code{qalys_}}{An object of class \link{qalys} simulated using \verb{$sim_qalys()}.}

\item{\code{costs_}}{An object of class \link{costs} simulated using \verb{$sim_costs()}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{CohortDtstm$new()}}
\item \href{#method-sim_stateprobs}{\code{CohortDtstm$sim_stateprobs()}}
\item \href{#method-sim_qalys}{\code{CohortDtstm$sim_qalys()}}
\item \href{#method-sim_costs}{\code{CohortDtstm$sim_costs()}}
\item \href{#method-summarize}{\code{CohortDtstm$summarize()}}
\item \href{#method-clone}{\code{CohortDtstm$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{CohortDtstm} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$new(trans_model = NULL, utility_model = NULL, cost_models = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{trans_model}}{The \code{trans_model} field.}

\item{\code{utility_model}}{The \code{utility_model} field.}

\item{\code{cost_models}}{The \code{cost_models} field.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{CohortDtstm} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sim_stateprobs"></a>}}
\if{latex}{\out{\hypertarget{method-sim_stateprobs}{}}}
\subsection{Method \code{sim_stateprobs()}}{
Simulate health state probabilities using \code{CohortDtstmTrans$sim_stateprobs()}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$sim_stateprobs(n_cycles)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{n_cycles}}{The number of model cycles to simulate the model for.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{stateprobs}
stored in \code{stateprobs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sim_qalys"></a>}}
\if{latex}{\out{\hypertarget{method-sim_qalys}{}}}
\subsection{Method \code{sim_qalys()}}{
Simulate quality-adjusted life-years (QALYs) as a function of \code{stateprobs_} and
\code{utility_model}. See \code{vignette("expected-values")} for details.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$sim_qalys(
  dr = 0.03,
  integrate_method = c("trapz", "riemann_left", "riemann_right"),
  lys = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{integrate_method}}{Method used to integrate state values when computing (QALYs).}

\item{\code{lys}}{If \code{TRUE}, then life-years are simulated in addition to QALYs.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{qalys} stored
in \code{qalys_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sim_costs"></a>}}
\if{latex}{\out{\hypertarget{method-sim_costs}{}}}
\subsection{Method \code{sim_costs()}}{
Simulate costs as a function of \code{stateprobs_} and \code{cost_models}.
See \code{vignette("expected-values")} for details.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$sim_costs(
  dr = 0.03,
  integrate_method = c("trapz", "riemann_left", "riemann_right")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{integrate_method}}{Method used to integrate state values when computing costs.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{costs} stored
in \code{costs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-summarize"></a>}}
\if{latex}{\out{\hypertarget{method-summarize}{}}}
\subsection{Method \code{summarize()}}{
Summarize costs and QALYs so that cost-effectiveness analysis can be performed.
See \code{\link[=summarize_ce]{summarize_ce()}}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$summarize(by_grp = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{by_grp}}{If \code{TRUE}, then costs and QALYs are computed by subgroup. If
\code{FALSE}, then costs and QALYs are aggregated across all patients (and subgroups).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
