% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/combinef.R
\name{combinef}
\alias{combinef}
\title{Optimally combine forecasts from a hierarchical or grouped time series}
\usage{
combinef(
  fcasts,
  nodes = NULL,
  groups = NULL,
  weights = NULL,
  nonnegative = FALSE,
  algorithms = c("lu", "cg", "chol", "recursive", "slm"),
  keep = c("gts", "all", "bottom"),
  parallel = FALSE,
  num.cores = 2,
  control.nn = list()
)
}
\arguments{
\item{fcasts}{Matrix of forecasts for all levels of the hierarchical time
series. Each row represents one forecast horizon and each column represents
one time series from the hierarchy.}

\item{nodes}{If the object class is \code{hts}, a list contains the number
of child nodes referring to \code{hts}.}

\item{groups}{If the object class is \code{gts}, a gmatrix is required,
which is the same as \code{groups} in the function \code{gts}.}

\item{weights}{A numeric vector. The default is \code{NULL} which means that
ordinary least squares is implemented.}

\item{nonnegative}{Logical. Should the reconciled forecasts be non-negative?}

\item{algorithms}{An algorithm to be used for computing reconciled
forecasts. See \code{\link{forecast.gts}} for details.}

\item{keep}{Return a \code{gts} object or the the reconciled forecasts at
the bottom level.}

\item{parallel}{Logical. Import parallel package to allow parallel processing.}

\item{num.cores}{Numeric. Specify how many cores are going to be used.}

\item{control.nn}{A list of control parameters to be passed on to the
block principal pivoting algorithm. See 'Details'.}
}
\value{
Return the (non-negative) reconciled \code{gts} object or forecasts at the bottom
level.
}
\description{
Using the methods of Hyndman et al. (2016) and Hyndman et al. (2011), this function optimally combines
the forecasts at all levels of a hierarchical time series. The
\code{\link{forecast.gts}} calls this function when the \code{comb} method
is selected.
}
\details{
The \code{control.nn} argument is a list that can supply any of the following components:
\describe{
\item{\code{ptype}}{Permutation method to be used: \code{"fixed"} or \code{"random"}. Defaults to \code{"fixed"}.}
\item{\code{par}}{The number of full exchange rules that may be tried. Defaults to 10.}
\item{\code{gtol}}{The tolerance of the convergence criteria. Defaults to \code{sqrt(.Machine$double.eps)}.}
}
}
\examples{

# hts example
\dontrun{
h <- 12
ally <- aggts(htseg1)
allf <- matrix(NA, nrow = h, ncol = ncol(ally))
for(i in 1:ncol(ally))
	allf[,i] <- forecast(auto.arima(ally[,i]), h = h)$mean
allf <- ts(allf, start = 51)
y.f <- combinef(allf, get_nodes(htseg1), weights = NULL, keep = "gts", algorithms = "lu")
plot(y.f)
}

\dontrun{
h <- 12
ally <- abs(aggts(htseg2))
allf <- matrix(NA, nrow = h, ncol = ncol(ally))
for(i in 1:ncol(ally))
  allf[,i] <- forecast(auto.arima(ally[,i], lambda = 0, biasadj = TRUE), h = h)$mean
b.f <- combinef(allf, get_nodes(htseg2), weights = NULL, keep = "bottom",
algorithms = "lu")
b.nnf <- combinef(allf, get_nodes(htseg2), weights = NULL, keep = "bottom",
algorithms = "lu", nonnegative = TRUE)
}

# gts example
\dontrun{
abc <- ts(5 + matrix(sort(rnorm(200)), ncol = 4, nrow = 50))
g <- rbind(c(1,1,2,2), c(1,2,1,2))
y <- gts(abc, groups = g)
h <- 12
ally <- aggts(y)
allf <- matrix(NA,nrow = h,ncol = ncol(ally))
for(i in 1:ncol(ally))
  allf[,i] <- forecast(auto.arima(ally[,i]),h = h)$mean
allf <- ts(allf, start = 51)
y.f <- combinef(allf, groups = get_groups(y), keep ="gts", algorithms = "lu")
plot(y.f)
}
}
\references{
Hyndman, R. J., Ahmed, R. A., Athanasopoulos, G., & Shang, H. L.
(2011). Optimal combination forecasts for hierarchical time series.
\emph{Computational Statistics and Data Analysis}, \bold{55}(9), 2579--2589. \url{https://robjhyndman.com/publications/hierarchical/}

Hyndman, R. J., Lee, A., & Wang, E. (2016). Fast computation of reconciled
forecasts for hierarchical and grouped time series. \emph{Computational Statistics and Data Analysis},
\bold{97}, 16--32. \url{https://robjhyndman.com/publications/hgts/}

Wickramasuriya, S. L., Turlach, B. A., & Hyndman, R. J. (to appear). Optimal non-negative forecast reconciliation.
\emph{Statistics and Computing}. \url{https://robjhyndman.com/publications/nnmint/}
}
\seealso{
\code{\link[hts]{hts}}, \code{\link[hts]{forecast.gts}}
}
\author{
Alan Lee, Rob J Hyndman, Earo Wang and Shanika L Wickramasuriya
}
\keyword{ts}
