% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multi-biasCorrect(generic).R
\name{applyBiasFactor}
\alias{applyBiasFactor}
\alias{applyBiasFactor,data.frame,biasFactor-method}
\alias{applyBiasFactor,list,biasFactor.hyfo-method}
\title{Apply bias factor to different forecasts for multi/operational/real time bias correction.}
\usage{
applyBiasFactor(frc, biasFactor, obs = NULL)

\S4method{applyBiasFactor}{data.frame,biasFactor}(frc, biasFactor, obs = NULL)

\S4method{applyBiasFactor}{list,biasFactor.hyfo}(frc, biasFactor, obs = NULL)
}
\arguments{
\item{frc}{a hyfo grid data output or a dataframe(time series) consists of Date column and one or more value columns, 
representing the frc data. Check details for more information.}

\item{biasFactor}{a file containing all the information of the calibration, will be
applied to different forecasts.}

\item{obs}{for some methods, observation input is necessary. obs is a hyfo grid data output or a dataframe (time series) consists of Date column and one or more value columns, 
representing the observation data. Default value is NULL.}
}
\description{
When you do multi/operational/real time bias correction. It's too expensive
to input hindcast and obs every time. Especially when you have a long period of hindcast
and obs, but only a short period of frc, it's too unecessary to read and compute hindcast
and obs everytime. Therefore, biasFactor is designed. Using \code{getBiasFactor}, you can
get the biasFactor with hindcast and observation, then you can use \code{applyBiasFactor} to 
apply the biasFactor to different forecasts.
}
\details{
Information about the method and how biasCorrect works can be found in \code{\link{biasCorrect}}

\strong{why use biasFactor}

As for forecasting, for daily data, there is usually no need to have
different bias factor every different day. You can calculate one bisa factor using a long
period of hindcast and obs, and apply that factor to different frc.

For example,

You have 10 years of hindcast and observation. you want to do bias correction for some 
forecasting product, e.g. system 4. For system 4, each month, you will get a new forecast
about the future 6 months. So if you want to do the real time bias correction, you have to
take the 10 years of hindcast and observation data with you, and run \code{biasCorrect} every
time you get a new forecast. That's too expensive.

For some practical use in forecasting, there isn't a so high demand for accuracy. E.g.,
Maybe for February and March, you can use the same biasFactor, no need to do the computation 
again. 


It is a generic function, if in your case you need to debug, please see \code{?debug()} 
for how to debug S4 method.
}
\examples{

######## hyfo grid file biascorrection
########

# If your input is obtained by \code{loadNcdf}, you can also directly biascorrect
# the file.

# First load ncdf file.
filePath <- system.file("extdata", "tnc.nc", package = "hyfo")
varname <- getNcdfVar(filePath)    
nc <- loadNcdf(filePath, varname)

data(tgridData)
#' # Since the example data, has some NA values, the process will include some warning #message, 
# which can be ignored in this case.



# Then we will use nc data as forecasting data, and use itself as hindcast data,
# use tgridData as observation.

biasFactor <- getBiasFactor(nc, tgridData)
newFrc <- applyBiasFactor(nc, biasFactor)
   
biasFactor <- getBiasFactor(nc, tgridData, method = 'eqm', extrapolate = 'constant',
preci = TRUE)
# This method needs obs input.
newFrc <- applyBiasFactor(nc, biasFactor, obs = tgridData)

biasFactor <- getBiasFactor(nc, tgridData, method = 'gqm', preci = TRUE)
newFrc <- applyBiasFactor(nc, biasFactor) 


######## Time series biascorrection
########

# Use the time series from testdl as an example, we take frc, hindcast and obs from testdl.
data(testdl)

# common period has to be extracted in order to better train the forecast.

datalist <- extractPeriod(testdl, startDate = '1994-1-1', endDate = '1995-10-1')

frc <- datalist[[1]]
hindcast <- datalist[[2]]
obs <- datalist[[3]]


# The data used here is just for example, so there could be negative data.

# default method is scaling
biasFactor <- getBiasFactor(hindcast, obs)
frc_new <- applyBiasFactor(frc, biasFactor)

# for precipitation data, extra process needs to be executed, so you have to tell
# the program to it is a precipitation data.

biasFactor <- getBiasFactor(hindcast, obs, preci = TRUE)
frc_new1 <- applyBiasFactor(frc, biasFactor)

# You can use other methods to biascorrect, e.g. delta method. 
biasFactor <- getBiasFactor(hindcast, obs, method = 'delta')
# delta method needs obs input.
frc_new2 <- applyBiasFactor(frc, biasFactor, obs = obs)

# 
biasFactor <- getBiasFactor(hindcast, obs, method = 'eqm', preci = TRUE)
# eqm needs obs input
frc_new3 <- applyBiasFactor(frc, biasFactor, obs = obs)

biasFactor <- getBiasFactor(hindcast, obs, method = 'gqm', preci = TRUE)
frc_new4 <- applyBiasFactor(frc, biasFactor)

plotTS(obs, frc, frc_new, frc_new1, frc_new2, frc_new3, frc_new4, plot = 'cum')

# You can also give name to this input list.
TSlist <- list(obs, frc, frc_new, frc_new1, frc_new2, frc_new3, frc_new4)
names(TSlist) <- c('obs', 'frc', 'delta', 'delta_preci', 'scale', 'eqm', 'gqm')
plotTS(list = TSlist, plot = 'cum')


# If the forecasts you extracted only has incontinuous data for certain months and years, e.g.,
# for seasonal forecasting, forecasts only provide 3-6 months data, so the case can be 
# for example Dec, Jan and Feb of every year from year 1999-2005.
# In such case, you need to extract certain months and years from observed time series.
# extractPeriod() can be then used.
  
  



# More examples can be found in the user manual on https://yuanchao-xu.github.io/hyfo/


}
\references{
Bias correction methods come from \code{biasCorrection} from \code{dowscaleR}

\itemize{

\item Santander Meteorology Group (2015). downscaleR: Climate data manipulation and statistical downscaling. R
package version 0.6-0. https://github.com/SantanderMetGroup/downscaleR/wiki

\item R.A.I. Wilcke, T. Mendlik and A. Gobiet (2013) Multi-variable error correction of regional climate models. Climatic Change, 120, 871-887

\item A. Amengual, V. Homar, R. Romero, S. Alonso, and C. Ramis (2012) A Statistical Adjustment of Regional Climate Model Outputs to Local Scales: Application to Platja de Palma, Spain. J. Clim., 25, 939-957

\item C. Piani, J. O. Haerter and E. Coppola (2009) Statistical bias correction for daily precipitation in regional climate models over Europe, Theoretical and Applied Climatology, 99, 187-192

\item O. Gutjahr and G. Heinemann (2013) Comparing precipitation bias correction methods for high-resolution regional climate simulations using COSMO-CLM, Theoretical and Applied Climatology, 114, 511-529
}
}
\seealso{
\code{\link{biasCorrect}} for method used in bias correction. 
\code{\link{getBiasFactor}}, for the first part.
}
\author{
Yuanchao Xu \email{xuyuanchao37@gmail.com }
}
