% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sat_vapor_pressure.R
\encoding{UTF-8}
\name{density_water}
\alias{density_water}
\title{Density of Saturated Liquid Water}
\usage{
density_water(
  Temp,
  units = c("SI", "Eng", "Absolute"),
  Eng_units = c("slug/ft^3", "lbm/ft^3")
)
}
\arguments{
\item{Temp}{numeric vector that contains the temperature (degrees Celsius,
degrees Fahrenheit, or Kelvin)}

\item{units}{character vector that contains the system of units {options are
\code{SI} for International System of Units, \code{Eng} for English units
(United States Customary System in the United States and Imperial Units in
the United Kingdom), or \code{Absolute} for Absolute Units}}

\item{Eng_units}{character vector that contains the unit for the density of
water {options are slug/ft^3 or lbm/ft^3}}
}
\value{
the density as a numeric vector. The units are not returned.
}
\description{
This function solves for the density of water using only the temperature of
the water in either units of degrees Celsius, degrees Fahrenheit, or Kelvin.
}
\details{
The simplified equation is expressed as

\deqn{\frac{\\rho^{t}}{\\rho_c} = 1 + b_1 \times \\tau^{1/3} + b_2 \times \\tau^{2/3} + b_3 \times \\tau^{5/3} + b_4 \times \\tau^{16/3} + b_5 \times \\tau^{43/3} + b_6 \times \\tau^{110/3}}

where \deqn{\\rho_c = 322 \frac{kg}{m^3}}

where \deqn{\\tau = 1 - \\theta}

where \deqn{\\theta = \frac{T}{T_c}}

where \deqn{T_c = 647.096 K}

with

\deqn{b_1 = 1.99274064}
\deqn{b_2 = 1.09965342}
\deqn{b_3 = -0.510839303}
\deqn{b_4 = -1.75493479}
\deqn{b_5 = -45.5170352}
\deqn{b_6 = -6.74694450 * 10 ^ 5}

\describe{
  \item{\emph{\\rho' = rho^t in the equation}}{Water Density (mass divided by volume) [kg/m^3, slug/ft^3, or lbm/ft^3]}
  \item{\emph{\\rho_c}}{Water Density at the critical point, kg/m^3}
  \item{\emph{T}}{the water temperature, Kelvin}
  \item{\emph{T_c}}{the critical water temperature, Kelvin}
}
}
\note{
Note: 1 lbf = 1 slug * 1 ft/sec^2, thus 1 slug = 1 lbf * sec^2 / 1 ft
(Reference 2)

Thus, lbm/ft^3 = lbf*s^2/ft/ft^3
}
\examples{
# Example 1 (Compare to reference standard in Reference paper)

library(iemisc)

273.16 # K

373.1243 # K

647.096 # K

Temp <- c(273.16, 373.1243, 647.096)

round::round_r3(density_water(Temp, units = "Absolute"), d = 3)


# Reference standard

999.789 # kg/m^3
958.365 # kg/m^3
322 # kg/m^3





# Example 2 - Example from the hydraulics package

library(iemisc)

rho <- hydraulics::dens(T = 25, units = "SI"); rho

rho2 <- density_water(Temp = 25, units = "SI"); rho2





# Example 3 - compare with densityH2Ov from aiRthermo

install.load::load_package("iemisc", "units")

Temp <- 180

# create a numeric vector with the units of degrees Celsius
T_C <- set_units(Temp, "degree_C")
T_C

# create a numeric vector to convert from degrees Celsius to Kelvin
T_K <- T_C
T_K

# create a numeric vector with the units of Kelvin
units(T_K) <- make_units(K)

pre <- aiRthermo::saturation_pressure_H2O(drop_units(T_K))
pre

rho_h2o <- aiRthermo::densityH2Ov(pre, drop_units(T_K), consts =
aiRthermo::export_constants()); rho_h2o

# Should not be the same as aiRthermo deals with water vapor rather than
# saturated liquid water

density_water(Temp = drop_units(T_K), units = "Absolute")





}
\references{
IAPWS SR1-86 (1992). "Revised Supplementary Release on Saturation Properties of Ordinary Water Substance". September 1992, \url{http://www.iapws.org/relguide/Supp-sat.html}
}
\author{
Irucka Embry
}
