% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/paths.R, R/structural.properties.R
\name{distance_table}
\alias{distance_table}
\alias{mean_distance}
\alias{distances}
\alias{shortest.paths}
\alias{get.shortest.paths}
\alias{get.all.shortest.paths}
\alias{average.path.length}
\alias{path.length.hist}
\alias{all_shortest_paths}
\alias{shortest_paths}
\title{Shortest (directed or undirected) paths between vertices}
\usage{
distance_table(graph, directed = TRUE)

mean_distance(
  graph,
  weights = NULL,
  directed = TRUE,
  unconnected = TRUE,
  details = FALSE
)

distances(
  graph,
  v = V(graph),
  to = V(graph),
  mode = c("all", "out", "in"),
  weights = NULL,
  algorithm = c("automatic", "unweighted", "dijkstra", "bellman-ford", "johnson")
)

shortest_paths(
  graph,
  from,
  to = V(graph),
  mode = c("out", "all", "in"),
  weights = NULL,
  output = c("vpath", "epath", "both"),
  predecessors = FALSE,
  inbound.edges = FALSE,
  algorithm = c("automatic", "unweighted", "dijkstra", "bellman-ford")
)

all_shortest_paths(
  graph,
  from,
  to = V(graph),
  mode = c("out", "all", "in"),
  weights = NULL
)
}
\arguments{
\item{graph}{The graph to work on.}

\item{directed}{Whether to consider directed paths in directed graphs,
this argument is ignored for undirected graphs.}

\item{weights}{Possibly a numeric vector giving edge weights. If this is
\code{NULL} and the graph has a \code{weight} edge attribute, then the
attribute is used. If this is \code{NA} then no weights are used (even if
the graph has a \code{weight} attribute).}

\item{unconnected}{What to do if the graph is unconnected (not
strongly connected if directed paths are considered). If TRUE, only
the lengths of the existing paths are considered and averaged; if
FALSE, the length of the missing paths are considered as having infinite
length, making the mean distance infinite as well.}

\item{details}{Whether to provide additional details in the result.
Functions accepting this argument (like \code{mean_distance()}) return
additional information like the number of disconnected vertex pairs in
the result when this parameter is set to \code{TRUE}.}

\item{v}{Numeric vector, the vertices from which the shortest paths will be
calculated.}

\item{to}{Numeric vector, the vertices to which the shortest paths will be
calculated. By default it includes all vertices. Note that for
\code{distances()} every vertex must be included here at most once. (This
is not required for \code{shortest_paths()}.}

\item{mode}{Character constant, gives whether the shortest paths to or from
the given vertices should be calculated for directed graphs. If \code{out}
then the shortest paths \emph{from} the vertex, if \verb{in} then \emph{to}
it will be considered. If \code{all}, the default, then the corresponding
undirected graph will be used, i.e. not directed paths are searched. This
argument is ignored for undirected graphs.}

\item{algorithm}{Which algorithm to use for the calculation. By default
igraph tries to select the fastest suitable algorithm. If there are no
weights, then an unweighted breadth-first search is used, otherwise if all
weights are positive, then Dijkstra's algorithm is used. If there are
negative weights and we do the calculation for more than 100 sources, then
Johnson's algorithm is used. Otherwise the Bellman-Ford algorithm is used.
You can override igraph's choice by explicitly giving this parameter. Note
that the igraph C core might still override your choice in obvious cases,
i.e. if there are no edge weights, then the unweighted algorithm will be
used, regardless of this argument.}

\item{from}{Numeric constant, the vertex from or to the shortest paths will
be calculated. Note that right now this is not a vector of vertex ids, but
only a single vertex.}

\item{output}{Character scalar, defines how to report the shortest paths.
\dQuote{vpath} means that the vertices along the paths are reported, this
form was used prior to igraph version 0.6. \dQuote{epath} means that the
edges along the paths are reported. \dQuote{both} means that both forms are
returned, in a named list with components \dQuote{vpath} and \dQuote{epath}.}

\item{predecessors}{Logical scalar, whether to return the predecessor vertex
for each vertex. The predecessor of vertex \code{i} in the tree is the
vertex from which vertex \code{i} was reached. The predecessor of the start
vertex (in the \code{from} argument) is itself by definition. If the
predecessor is zero, it means that the given vertex was not reached from the
source during the search. Note that the search terminates if all the
vertices in \code{to} are reached.}

\item{inbound.edges}{Logical scalar, whether to return the inbound edge for
each vertex. The inbound edge of vertex \code{i} in the tree is the edge via
which vertex \code{i} was reached. The start vertex and vertices that were
not reached during the search will have zero in the corresponding entry of
the vector. Note that the search terminates if all the vertices in \code{to}
are reached.}
}
\value{
For \code{distances()} a numeric matrix with \code{length(to)}
columns and \code{length(v)} rows. The shortest path length from a vertex to
itself is always zero. For unreachable vertices \code{Inf} is included.

For \code{shortest_paths()} a named list with four entries is returned:
\item{vpath}{This itself is a list, of length \code{length(to)}; list
element \code{i} contains the vertex ids on the path from vertex \code{from}
to vertex \code{to[i]} (or the other way for directed graphs depending on
the \code{mode} argument). The vector also contains \code{from} and \code{i}
as the first and last elements. If \code{from} is the same as \code{i} then
it is only included once. If there is no path between two vertices then a
numeric vector of length zero is returned as the list element. If this
output is not requested in the \code{output} argument, then it will be
\code{NULL}.} \item{epath}{This is a list similar to \code{vpath}, but the
vectors of the list contain the edge ids along the shortest paths, instead
of the vertex ids. This entry is set to \code{NULL} if it is not requested
in the \code{output} argument.} \item{predecessors}{Numeric vector, the
predecessor of each vertex in the \code{to} argument, or \code{NULL} if it
was not requested.} \item{inbound_edges}{Numeric vector, the inbound edge
for each vertex, or \code{NULL}, if it was not requested.}

For \code{all_shortest_paths()} a list is returned, each list element
contains a shortest path from \code{from} to a vertex in \code{to}. The
shortest paths to the same vertex are collected into consecutive elements of
the list.

For \code{mean_distance()} a single number is returned if \code{details=FALSE},
or a named list with two entries: \code{res} is the mean distance as a numeric
scalar and \code{unconnected} is the number of unconnected vertex pairs,
also as a numeric scalar.

\code{distance_table()} returns a named list with two entries: \code{res} is
a numeric vector, the histogram of distances, \code{unconnected} is a
numeric scalar, the number of pairs for which the first vertex is not
reachable from the second. The sum of the two entries is always \eqn{n(n-1)}
for directed graphs and \eqn{n(n-1)/2} for undirected graphs.
}
\description{
\code{distances()} calculates the length of all the shortest paths from
or to the vertices in the network. \code{shortest_paths()} calculates one
shortest path (the path itself, and not just its length) from or to the
given vertex.
}
\details{
The shortest path, or geodesic between two pair of vertices is a path with
the minimal number of vertices. The functions documented in this manual page
all calculate shortest paths between vertex pairs.

\code{distances()} calculates the lengths of pairwise shortest paths from
a set of vertices (\code{from}) to another set of vertices (\code{to}). It
uses different algorithms, depending on the \code{algorithm} argument and
the \code{weight} edge attribute of the graph. The implemented algorithms
are breadth-first search (\sQuote{\code{unweighted}}), this only works for
unweighted graphs; the Dijkstra algorithm (\sQuote{\code{dijkstra}}), this
works for graphs with non-negative edge weights; the Bellman-Ford algorithm
(\sQuote{\code{bellman-ford}}), and Johnson's algorithm
(\sQuote{\code{johnson}}). The latter two algorithms work with arbitrary
edge weights, but (naturally) only for graphs that don't have a negative
cycle. Note that a negative-weight edge in an undirected graph implies
such a cycle. Johnson's algorithm performs better than the Bellman-Ford
one when many source (and target) vertices are given, with all-pairs
shortest path length calculations being the typical use case.

igraph can choose automatically between algorithms, and chooses the most
efficient one that is appropriate for the supplied weights (if any). For
automatic algorithm selection, supply \sQuote{\code{automatic}} as the
\code{algorithm} argument. (This is also the default.)

\code{shortest_paths()} calculates a single shortest path (i.e. the path
itself, not just its length) between the source vertex given in \code{from},
to the target vertices given in \code{to}. \code{shortest_paths()} uses
breadth-first search for unweighted graphs and Dijkstra's algorithm for
weighted graphs. The latter only works if the edge weights are non-negative.

\code{all_shortest_paths()} calculates \emph{all} shortest paths between
pairs of vertices. More precisely, between the \code{from} vertex to the
vertices given in \code{to}. It uses a breadth-first search for unweighted
graphs and Dijkstra's algorithm for weighted ones. The latter only supports
non-negative edge weights.

\code{mean_distance()} calculates the average path length in a graph, by
calculating the shortest paths between all pairs of vertices (both ways for
directed graphs). It uses a breadth-=first search for unweighted graphs and
Dijkstra's algorithm for weighted ones. The latter only supports non-negative
edge weights.

\code{distance_table()} calculates a histogram, by calculating the shortest
path length between each pair of vertices. For directed graphs both
directions are considered, so every pair of vertices appears twice in the
histogram.
}
\examples{

g <- make_ring(10)
distances(g)
shortest_paths(g, 5)
all_shortest_paths(g, 1, 6:8)
mean_distance(g)
## Weighted shortest paths
el <- matrix(
  ncol = 3, byrow = TRUE,
  c(
    1, 2, 0,
    1, 3, 2,
    1, 4, 1,
    2, 3, 0,
    2, 5, 5,
    2, 6, 2,
    3, 2, 1,
    3, 4, 1,
    3, 7, 1,
    4, 3, 0,
    4, 7, 2,
    5, 6, 2,
    5, 8, 8,
    6, 3, 2,
    6, 7, 1,
    6, 9, 1,
    6, 10, 3,
    8, 6, 1,
    8, 9, 1,
    9, 10, 4
  )
)
g2 <- add_edges(make_empty_graph(10), t(el[, 1:2]), weight = el[, 3])
distances(g2, mode = "out")

}
\references{
West, D.B. (1996). \emph{Introduction to Graph Theory.} Upper
Saddle River, N.J.: Prentice Hall.
}
\seealso{
Other paths: 
\code{\link{all_simple_paths}()},
\code{\link{diameter}()},
\code{\link{eccentricity}()},
\code{\link{radius}()}

Other structural.properties: 
\code{\link{bfs}()},
\code{\link{component_distribution}()},
\code{\link{connect}()},
\code{\link{constraint}()},
\code{\link{coreness}()},
\code{\link{degree}()},
\code{\link{dfs}()},
\code{\link{edge_density}()},
\code{\link{feedback_arc_set}()},
\code{\link{girth}()},
\code{\link{is_dag}()},
\code{\link{is_matching}()},
\code{\link{knn}()},
\code{\link{laplacian_matrix}()},
\code{\link{reciprocity}()},
\code{\link{subcomponent}()},
\code{\link{subgraph}()},
\code{\link{topo_sort}()},
\code{\link{transitivity}()},
\code{\link{unfold_tree}()},
\code{\link{which_multiple}()},
\code{\link{which_mutual}()}

Other structural.properties: 
\code{\link{bfs}()},
\code{\link{component_distribution}()},
\code{\link{connect}()},
\code{\link{constraint}()},
\code{\link{coreness}()},
\code{\link{degree}()},
\code{\link{dfs}()},
\code{\link{edge_density}()},
\code{\link{feedback_arc_set}()},
\code{\link{girth}()},
\code{\link{is_dag}()},
\code{\link{is_matching}()},
\code{\link{knn}()},
\code{\link{laplacian_matrix}()},
\code{\link{reciprocity}()},
\code{\link{subcomponent}()},
\code{\link{subgraph}()},
\code{\link{topo_sort}()},
\code{\link{transitivity}()},
\code{\link{unfold_tree}()},
\code{\link{which_multiple}()},
\code{\link{which_mutual}()}

Other structural.properties: 
\code{\link{bfs}()},
\code{\link{component_distribution}()},
\code{\link{connect}()},
\code{\link{constraint}()},
\code{\link{coreness}()},
\code{\link{degree}()},
\code{\link{dfs}()},
\code{\link{edge_density}()},
\code{\link{feedback_arc_set}()},
\code{\link{girth}()},
\code{\link{is_dag}()},
\code{\link{is_matching}()},
\code{\link{knn}()},
\code{\link{laplacian_matrix}()},
\code{\link{reciprocity}()},
\code{\link{subcomponent}()},
\code{\link{subgraph}()},
\code{\link{topo_sort}()},
\code{\link{transitivity}()},
\code{\link{unfold_tree}()},
\code{\link{which_multiple}()},
\code{\link{which_mutual}()}

Other structural.properties: 
\code{\link{bfs}()},
\code{\link{component_distribution}()},
\code{\link{connect}()},
\code{\link{constraint}()},
\code{\link{coreness}()},
\code{\link{degree}()},
\code{\link{dfs}()},
\code{\link{edge_density}()},
\code{\link{feedback_arc_set}()},
\code{\link{girth}()},
\code{\link{is_dag}()},
\code{\link{is_matching}()},
\code{\link{knn}()},
\code{\link{laplacian_matrix}()},
\code{\link{reciprocity}()},
\code{\link{subcomponent}()},
\code{\link{subgraph}()},
\code{\link{topo_sort}()},
\code{\link{transitivity}()},
\code{\link{unfold_tree}()},
\code{\link{which_multiple}()},
\code{\link{which_mutual}()}
}
\author{
Gabor Csardi \email{csardi.gabor@gmail.com}
}
\concept{paths}
\concept{structural.properties}
\keyword{graphs}
