% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/micro_read_yield.R
\name{read_ipums_micro_yield}
\alias{read_ipums_micro_yield}
\alias{read_ipums_micro_list_yield}
\alias{IpumsLongYield}
\alias{IpumsListYield}
\title{Read data from an IPUMS microdata extract in yields}
\usage{
read_ipums_micro_yield(
  ddi,
  vars = NULL,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)

read_ipums_micro_list_yield(
  ddi,
  vars = NULL,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)
}
\arguments{
\item{ddi}{Either a path to a DDI .xml file downloaded from
\href{https://www.ipums.org/}{IPUMS}, or an
\link{ipums_ddi} object parsed by \code{\link[=read_ipums_ddi]{read_ipums_ddi()}}. See
\emph{Downloading IPUMS files} below.}

\item{vars}{Names of variables to include in the output. Accepts a
vector of names or a \link[=selection_language]{tidyselect selection}.
If \code{NULL}, includes all variables in the file.

For hierarchical data, the \code{RECTYPE} variable is always included even if
unspecified.}

\item{data_file}{Path to the data (.gz) file associated with
the provided \code{ddi} file. By default, looks for the data file in the same
directory as the DDI file. If the data file has been moved, specify
its location here.}

\item{verbose}{Logical indicating whether to display IPUMS conditions and
progress information.}

\item{var_attrs}{Variable attributes from the DDI to add to the columns of
the output data. Defaults to all available attributes.
See \code{\link[=set_ipums_var_attributes]{set_ipums_var_attributes()}} for more details.}

\item{lower_vars}{If reading a DDI from a file,
a logical indicating whether to convert variable names to lowercase.
Defaults to \code{FALSE} for consistency with IPUMS conventions.

This argument will be ignored if argument \code{ddi} is
an \link{ipums_ddi} object. Use \code{\link[=read_ipums_ddi]{read_ipums_ddi()}} to convert variable
names to lowercase when reading a DDI file.

If \code{lower_vars = TRUE} and \code{vars} is specified, \code{vars} should reference the
lowercase column names.}
}
\value{
A HipYield R6 object (see Details section)
}
\description{
Read a microdata dataset downloaded from the IPUMS extract system into an
object that can read and operate on a group ("yield") of lines at a time.
Use these functions to read a file that is too large to store in memory at
a single time. They represent a more flexible implementation of
\code{\link[=read_ipums_micro_chunked]{read_ipums_micro_chunked()}} using R6.

Two files are required to load IPUMS microdata extracts:
\itemize{
\item A \href{https://ddialliance.org/learn/what-is-ddi}{DDI codebook} file
(.xml) used to parse the extract's data file
\item A data file (either .dat.gz or .csv.gz)
}

See \emph{Downloading IPUMS files} below for more information about downloading
these files.

\code{read_ipums_micro_yield()} and \code{read_ipums_micro_list_yield()} differ
in their handling of extracts that contain multiple record types.
See \emph{Data structures} below.

Note that these functions only support fixed-width (.dat) data files.
}
\section{Methods summary:}{
These functions return a HipYield R6 object with the following methods:
\itemize{
\item \code{yield(n = 10000)} reads the next "yield" from the
data.

For \code{read_ipums_micro_yield()}, returns a \code{\link[tibble:tbl_df-class]{tibble}}
with up to \code{n} rows.

For \code{read_ipums_micro_list_yield()}, returns a list of tibbles with a
total of up to \code{n} rows across list elements.

If fewer than \code{n} rows are left in the data, returns all remaining rows.
If no rows are left in the data, returns \code{NULL}.
\item \code{reset()} resets the data so that the next yield will read data from the
start.
\item \code{is_done()} returns a logical indicating whether all rows in the file
have been read.
\item \code{cur_pos} contains the next row number that will be read (1-indexed).
}
}

\section{Data structures}{


Files from IPUMS projects that contain data for multiple types of records
(e.g. household records and person records) may be either rectangular
or hierarchical.

Rectangular data are transformed such that each row of data
represents only one type of record. For instance, each row will represent
a person record, and all household-level information for that person will
be included in the same row.

Hierarchical data have records of
different types interspersed in a single file. For instance, a household
record will be included in its own row followed by the person records
associated with that household.

Hierarchical data can be read in two different formats:
\itemize{
\item \code{read_ipums_micro_yield()} produces an object that yields data as a
\code{\link[tibble:tbl_df-class]{tibble}} whose rows
represent single records, regardless of record type. Variables that do
not apply to a particular record type will be filled with \code{NA} in rows of
that record type. For instance, a person-specific variable will be missing
in all rows associated with household records.
\item \code{read_ipums_micro_list_yield()} produces an object that yields data as a
list of \code{tibble} objects, where each list element contains
only one record type. Each list element is named with its corresponding
record type. In this case, when using \code{yield()}, \code{n} refers to
the total number of rows \emph{across} record types, rather than in each
record type.
}
}

\section{Downloading IPUMS files}{


You must download both the DDI codebook and the data file from the IPUMS
extract system to load the data into R. \verb{read_ipums_micro_*()} functions
assume that the data file and codebook share a common base file name and
are present in the same directory. If this is not the case, provide a
separate path to the data file with the \code{data_file} argument.

If using the IPUMS extract interface:
\itemize{
\item Download the data file by clicking \strong{Download .dat} under
\strong{Download Data}.
\item Download the DDI codebook by right clicking on the \strong{DDI} link in the
\strong{Codebook} column of the extract interface and selecting \strong{Save as...}
(on Safari, you may have to select \strong{Download Linked File as...}).
Be sure that the codebook is downloaded in .xml format.
}

If using the IPUMS API:
\itemize{
\item For supported collections, use \code{\link[=download_extract]{download_extract()}} to download a completed
extract via the IPUMS API. This automatically downloads both the DDI
codebook and the data file from the extract and
returns the path to the codebook file.
}
}

\examples{
# Create an IpumsLongYield object
long_yield <- read_ipums_micro_yield(ipums_example("cps_00157.xml"))

# Yield the first 10 rows of the data
long_yield$yield(10)

# Yield the next 20 rows of the data
long_yield$yield(20)

# Check the current position after yielding 30 rows
long_yield$cur_pos

# Reset to the beginning of the file
long_yield$reset()

# Use a loop to flexibly process the data in pieces. Count all Minnesotans:
total_mn <- 0

while (!long_yield$is_done()) {
  cur_data <- long_yield$yield(1000)
  total_mn <- total_mn + sum(as_factor(cur_data$STATEFIP) == "Minnesota")
}

total_mn

# Can also read hierarchical data as list:
list_yield <- read_ipums_micro_list_yield(ipums_example("cps_00159.xml"))

# Yield size is based on total rows for all list elements
list_yield$yield(10)
}
\seealso{
\code{\link[=read_ipums_micro_chunked]{read_ipums_micro_chunked()}} to read data from large IPUMS
microdata extracts in chunks.

\code{\link[=read_ipums_micro]{read_ipums_micro()}} to read data from an IPUMS microdata extract.

\code{\link[=read_ipums_ddi]{read_ipums_ddi()}} to read metadata associated with an IPUMS microdata
extract.

\code{\link[=read_ipums_sf]{read_ipums_sf()}} to read spatial data from an IPUMS extract.

\code{\link[=ipums_list_files]{ipums_list_files()}} to list files in an IPUMS extract.
}
