% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootSE.R
\name{bootSE}
\alias{bootSE}
\title{Standard errors via bootstrap for an \code{mjoint} object}
\usage{
bootSE(object, nboot = 100, ci = 0.95, use.mle = TRUE, verbose = FALSE,
  control = list(), progress = TRUE, ...)
}
\arguments{
\item{object}{an object inheriting from class \code{mjoint} for a joint model
of time-to-event and multivariate longitudinal data.}

\item{nboot}{the number of bootstrap samples. Default is \code{nboot=100}.}

\item{ci}{the confidence interval to be estimated using the
percentile-method. Default is \code{ci=0.95} for a 95\% confidence
interval.}

\item{use.mle}{logical: should the algorithm use the maximizer from the
converged model in \code{object} as initial values for coefficients in each
bootstrap iteration. Default is \code{use.mle=TRUE}.}

\item{verbose}{logical: if \code{TRUE}, the parameter estimates and other
convergence statistics are value are printed at each iteration of the MCEM
algorithm. Default is \code{FALSE}.}

\item{control}{a list of control values with components: \describe{

  \item{\code{nMC}}{integer: the initial number of Monte Carlo samples to be
  used for integration in the burn-in phase of the MCEM. Default is
  \code{nMC=}max(100, 50\emph{K}).}

  \item{\code{nMCscale}}{integer: the scale factor for the increase in Monte
  Carlo size when Monte Carlo has not reduced from the previous iteration.
  Default is \code{nMCscale=3}.}

  \item{\code{nMCmax}}{integer: the maximum number of Monte Carlo samples
  that the algorithm is allowed to reach. Default is \code{nMCmax=20000}.}

  \item{\code{burnin}}{integer: the number of iterations for burn-in phase of
  the optimization algorithm. It is computationally inefficient to use a
  large number of Monte Carlo samples early on until one is approximately
  near the maximum likelihood estimate. Default is
  \code{burnin=}100\emph{K}.}

  \item{\code{mcmaxIter}}{integer: the maximum number of MCEM algorithm
  iterations allowed. Default is \code{mcmaxIter=burnin+200}.}

  \item{\code{convCrit}}{character string: the convergence criterion to be
  used. See \strong{Details}.}

  \item{\code{gammaOpt}}{character string: by default (\code{gammaOpt='NR'}),
  \eqn{\gamma} is updated using a one-step Newton-Raphson iteration, with the
  Hessian matrix calculated exactly. If \code{gammaOpt='GN'}, a Gauss-Newton
  algorithm-type iteration is implemented, where the Hessian matrix is
  approximated based on calculations similar to those used for calculating
  the empirical information matrix? If it is used, then the step-length is
  adjusted by a nominal scaling parameter of 0.5 in order to reduce the
  chance of over-shooting the maximizer.}

  \item{\code{tol0}}{numeric: tolerance value for convergence in the
  parameters; see \strong{Details}. Default is \code{5e-03}.}

  \item{\code{tol1}}{numeric: tolerance value for convergence in the
  parameters; see \strong{Details}. Default is \code{1e-03}.}

  \item{\code{tol2}}{numeric: tolerance value for convergence in the
  parameters; see \strong{Details}. Default is \code{5e-03}.}

  \item{\code{tol.em}}{numeric: tolerance value for convergence in the
  multivariate linear mixed model (MV-LMM). When \eqn{K > 1}, the optimal
  initial parameters are those from the MV-LMM, which is estimated using a
  separate EM algorithm. Since both the E- and M-steps are available in
  closed-form, this algorithm convergences relatively rapidly with a high
  precision. Default is min(\code{1e-04}, \code{tol2}).}

  \item{\code{rav}}{numeric: threshold when using \code{convCrit='sas'} that
  applies absolute change (when \eqn{<}\code{rav}) or relative change (when
  \eqn{\ge}\code{rav}) criterion; see \strong{Details}. Default is
  \code{0.1}, which is an order of magnitude higher than the SAS
  implementation.}

  }}

\item{progress}{logical: should a progress bar be shown on the console to
indicate the percentage of bootstrap iterations completed? Default is
\code{progress=TRUE}.}

\item{...}{options passed to the \code{control} argument.}
}
\value{
An object of class \code{bootSE}.
}
\description{
This function takes a model fit from an \code{mjoint} object and
  calculates standard errors and confidence intervals for the main
  longitudinal and survival coefficient parameters, including the latent
  association parameters, using bootstrapping (Efron and Tibshirani, 2000).
}
\details{
Standard errors and confidence intervals are obtained by repeated
  fitting of the requisite joint model to bootstrap samples of the original
  longitudinal and time-to-event data. Note that bootstrap is done by
  sampling subjects, not individual records.
}
\note{
This function is computationally intensive. A dynamic progress bar is
  displayed showing the percentage of bootstrap models fitted.

  Due to random sampling, an \code{mjoint} model fitted to some bootstrap
  samples may not converge within the specified control parameter settings.
  The \code{bootSE} code discards any models that failed to converge when
  calculating the standard errors and confidence intervals. If a large
  proportion of models have failed to converge, it is likely that it will
  need to be refitted with changes to the \code{control} arguments.
}
\examples{
\dontrun{
# Fit a joint model with bivariate longitudinal outcomes

data(heart.valve)
hvd <- heart.valve[!is.na(heart.valve$log.grad) & !is.na(heart.valve$log.lvmi), ]

fit <- mjoint(
    formLongFixed = list("grad" = log.grad ~ time + sex + hs,
                         "lvmi" = log.lvmi ~ time + sex),
    formLongRandom = list("grad" = ~ 1 | num,
                          "lvmi" = ~ time | num),
    formSurv = Surv(fuyrs, status) ~ age,
    data = list(hvd, hvd),
    inits = list("gamma" = c(0.11, 1.51, 0.80)),
    timeVar = "time",
    verbose = TRUE)

fit.boot <- bootSE(fit, 50, use.mle = TRUE, control = list(
    burnin = 25, convCrit = "either",
    tol0 = 6e-03, tol2 = 6e-03, mcmaxIter = 60))
}
}
\references{
Efron B, Tibshirani R. \emph{An Introduction to the Bootstrap.} 2000; Boca
Raton, FL: Chapman & Hall/CRC.
}
\seealso{
\code{\link{mjoint}} for approximate standard errors.
}
\author{
Graeme L. Hickey (\email{graeme.hickey@liverpool.ac.uk})
}
\keyword{methods}
\keyword{multivariate}
\keyword{survival}
