context("Testing 'kde1d'")

set.seed(0)
n_sim <- 1e2
data_types <- c("unbounded", "left_boundary", "right_boundary",
                "two_boundaries", "discrete")
deg <- 0:2

scenarios <- expand.grid(data_types = data_types, deg = deg)
scenarios <- split(scenarios, seq_len(nrow(scenarios)))
fits <- lapply(
    scenarios,
    function(scenario) {
        xmin <- xmax <- NaN
        if (scenario$data_type == "unbounded") {
            x <- rnorm(n_sim)
        } else if (scenario$data_type == "left_boundary") {
            x <- rexp(n_sim)
            xmin <- 0
        }  else if (scenario$data_type == "right_boundary") {
            x <- -rexp(n_sim)
            xmax <- 0
        }  else if (scenario$data_type == "two_boundaries") {
            x <- runif(n_sim)
            xmin <- 0
            xmax <- 1
        } else {
            x <- ordered(rbinom(n_sim, size = 5, prob = 0.5),
                         levels = 0:5)
        }
        kde1d(x, xmin = xmin, xmax = xmax, deg = scenario$deg)
    }
)

test_that("detects wrong arguments", {
    x <- rnorm(n_sim)
    expect_error(kde1d(x, xmin = 0))
    expect_error(kde1d(x, xmax = 0))
    expect_error(kde1d(x, xmin = 10, xmax = -10))
    expect_error(kde1d(x, mult = 0))
    expect_error(kde1d(x, deg = 3))
    expect_error(kde1d(x, weights = list()))
    expect_error(kde1d(x, weights = 1:3))

    x <- ordered(rbinom(n_sim, size = 5, prob = 0.5), levels = 0:5)
    expect_error(kde1d(x, xmax = 0))
})

test_that("returns proper 'kde1d' object", {
    lapply(fits, function(x) expect_s3_class(x, "kde1d"))

    class_members <- c("grid_points", "values", "bw", "xmin", "xmax", "deg",
                       "edf", "loglik", "weights", "jitter_info", "var_name",
                       "nobs")
    lapply(fits, function(x) expect_identical(names(x), class_members))
})

test_that("d/p/r/h functions work", {
    n <- 50
    u <- runif(n)
    test_dpqr <- function(fit, sim) {
        sim <- data.frame(sim)
        is_jittered <- length(fit$jitter_info$i_disc) == 1
        if (is.nan(fit$xmax)) {
            xmax <- ifelse(is_jittered, fit$jitter_info$nu, Inf)
        } else {
            xmax <- fit$xmax
        }
        if (is.nan(fit$xmin)) {
            xmin <- ifelse(is_jittered, 0, -Inf)
        } else {
            xmin <- fit$xmin
        }
        expect_that(all(sim >= xmin), equals(TRUE))
        expect_that(all(sim <= xmax), equals(TRUE))
        expect_gte(max(dkde1d(sim, fit), 0), 0)
        expect_gte(max(pkde1d(sim, fit), 0), 0)
        expect_lte(max(pkde1d(sim, fit), 1), 1)
        expect_that(all(qkde1d(u, fit) >= xmin), equals(TRUE))
        expect_that(all(qkde1d(u, fit) <= xmax), equals(TRUE))
        if (!is.nan(fit$xmin)) {
            expect_equal(dkde1d(xmin - 1, fit), 0)
            expect_equal(pkde1d(xmin - 1, fit), 0)
        }

        if (!is.nan(fit$xmax)) {
            expect_equal(dkde1d(xmax + 1, fit), 0)
            expect_equal(pkde1d(xmax + 1, fit), 1)
        }
    }

    sims <- lapply(fits, function(x) rkde1d(n, x))
    mapply(test_dpqr, fits, sims)

    sim <- lapply(fits, function(x) rkde1d(n, x, quasi = TRUE))
    mapply(test_dpqr, fits, sims)
})

test_that("plot functions work", {

    test_plot <- function(fit) {
        expect_silent(plot(fit))
        if (length(fit$jitter_info$i_disc) == 1) {
            expect_error(lines(fit))
        } else {
            expect_silent(lines(fit))
        }
    }

    lapply(fits, test_plot)
})

test_that("other generics work", {

    test_other_generics <- function(fit) {
        expect_output(print(fit))
        expect_output(s <- summary(fit))
        expect_is(s, "numeric")
        expect_equal(length(s), 4)
        expect_silent(s <- logLik(fit))
        expect_is(s, "numeric")
    }

    lapply(fits, test_other_generics)
})

test_that("behavior for discrete data is consistent", {
    n <- 1e3
    x <- ordered(sample(5, n, TRUE), 1:5)
    fit <- kde1d(x)
    xx <- ordered(1:5, 1:5)
    expect_equal(dkde1d(1:5, fit), dkde1d(xx, fit))
    expect_equal(pkde1d(1:5, fit), pkde1d(xx, fit))
    expect_true(all(is.na(dkde1d(c(0, 6), fit))))
    expect_true(all(is.na(pkde1d(c(0, 6), fit))))
    expect_true(all(rkde1d(n, fit) %in% x))
})

test_that("estimates for discrete data are reasonable", {
    x <- ordered(sample(5, 1e5, TRUE), 1:5)
    fit <- kde1d(x)
    expect_true(all(abs(dkde1d(1:5, fit) - 0.2) < 0.1))
})

test_that("works with weights", {
    n_sim <- 1000
    x <- rnorm(n_sim)

    fit <- kde1d(x, weights = rep(1, n_sim))
    fit0 <- kde1d(x)
    expect_equal(dkde1d(x, fit), dkde1d(x, fit0))

    fit <- kde1d(x, weights = c(rep(1, n_sim / 2), rep(0, n_sim / 2)))
    fit0 <- kde1d(x[seq_len(n_sim / 2)])
    expect_equal(dkde1d(x, fit), dkde1d(x, fit0), tolerance = 0.01)
})