% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LearningRateSchedule.R
\name{LearningRateSchedule}
\alias{LearningRateSchedule}
\title{Define a custom \code{LearningRateSchedule} class}
\usage{
LearningRateSchedule(
  classname,
  call = NULL,
  initialize = NULL,
  get_config = NULL,
  ...,
  public = list(),
  private = list(),
  inherit = NULL,
  parent_env = parent.frame()
)
}
\arguments{
\item{classname}{String, the name of the custom class. (Conventionally, CamelCase).}

\item{call, initialize, get_config}{Recommended methods to implement. See description and details sections.}

\item{..., public}{Additional methods or public members of the custom class.}

\item{private}{Named list of R objects (typically, functions) to include in
instance private environments. \code{private} methods will have all the same
symbols in scope as public methods (See section "Symbols in Scope"). Each
instance will have it's own \code{private} environment. Any objects
in \code{private} will be invisible from the Keras framework and the Python
runtime.}

\item{inherit}{What the custom class will subclass. By default, the base keras class.}

\item{parent_env}{The R environment that all class methods will have as a grandparent.}
}
\value{
A function that returns \code{LearningRateSchedule} instances, similar to the
built-in \verb{learning_rate_schedule_*} family of functions.
}
\description{
Subclass the keras learning rate schedule base class.

You can use a learning rate schedule to modulate how the learning rate
of your optimizer changes over time.

Several built-in learning rate schedules are available, such as
\code{\link[=learning_rate_schedule_exponential_decay]{learning_rate_schedule_exponential_decay()}} or
\code{\link[=learning_rate_schedule_piecewise_constant_decay]{learning_rate_schedule_piecewise_constant_decay()}}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{lr_schedule <- learning_rate_schedule_exponential_decay(
  initial_learning_rate = 1e-2,
  decay_steps = 10000,
  decay_rate = 0.9
)
optimizer <- optimizer_sgd(learning_rate = lr_schedule)
}\if{html}{\out{</div>}}

A \code{LearningRateSchedule()} instance can be passed in as the \code{learning_rate}
argument of any optimizer.

To implement your own schedule object, you should implement the \code{call}
method, which takes a \code{step} argument (a scalar integer backend tensor, the
current training step count).
Note that \code{step} is 0-based (i.e., the first step is \code{0}).
Like for any other Keras object, you can also optionally
make your object serializable by implementing the \code{get_config()}
and \code{from_config()} methods.
}
\section{Example}{
\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_custom_learning_rate_schedule <- LearningRateSchedule(
  classname = "MyLRSchedule",
  initialize = function( initial_learning_rate) \{

    self$initial_learning_rate <- initial_learning_rate
  \},


  call = function(step) \{
    # note that `step` is a tensor
    # and call() will be traced via tf_function() or similar.

    str(step) # <KerasVariable shape=(), dtype=int64, path=SGD/iteration>

    # print 'step' every 1000 steps
    op_cond((step \%\% 1000) == 0,
            \\() \{tensorflow::tf$print(step); NULL\},
            \\() \{NULL\})
    self$initial_learning_rate / (step + 1)
  \}
)

optimizer <- optimizer_sgd(
  learning_rate = my_custom_learning_rate_schedule(0.1)
)

# You can also call schedule instances directly
# (e.g., for interactive testing, or if implementing a custom optimizer)
schedule <- my_custom_learning_rate_schedule(0.1)
step <- keras$Variable(initializer = op_ones,
                       shape = shape(),
                       dtype = "int64")
schedule(step)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## <KerasVariable shape=(), dtype=int64, path=variable>

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.0, shape=(), dtype=float64)

}\if{html}{\out{</div>}}
}

\section{Methods available:}{
\itemize{
\item

\if{html}{\out{<div class="sourceCode">}}\preformatted{get_config()
}\if{html}{\out{</div>}}
}
}

\section{Symbols in scope}{
All R function custom methods (public and private) will have the
following symbols in scope:
\itemize{
\item \code{self}: The custom class instance.
\item \code{super}: The custom class superclass.
\item \code{private}: An R environment specific to the class instance.
Any objects assigned here are invisible to the Keras framework.
\item \verb{__class__} and \code{as.symbol(classname)}: the custom class type object.
}
}

\seealso{
Other optimizer learning rate schedules: \cr
\code{\link{learning_rate_schedule_cosine_decay}()} \cr
\code{\link{learning_rate_schedule_cosine_decay_restarts}()} \cr
\code{\link{learning_rate_schedule_exponential_decay}()} \cr
\code{\link{learning_rate_schedule_inverse_time_decay}()} \cr
\code{\link{learning_rate_schedule_piecewise_constant_decay}()} \cr
\code{\link{learning_rate_schedule_polynomial_decay}()} \cr
}
\concept{optimizer learning rate schedules}
