\name{plot}
\alias{plot.kde}
\alias{plot.kda.kde}

\title{Kernel density estimate plot for 1- to 3-dimensional data}
\description{
  Kernel density estimate plot for 1- to 3-dimensional data.
}

\usage{
\method{plot}{kde}(x, ...)

\method{plot}{kda.kde}(x, y, y.group, ...)
}

\arguments{
  \item{x}{an object of class \code{kde} (output from \code{\link{kde}}) or 
    an object of class \code{kda.kde} (output from \code{\link{kda.kde}})}
  \item{y}{matrix of test data points}
  \item{y.group}{vector of group labels for test data points}
  \item{...}{other graphics parameters}
}
  

\value{
  Plot of 1-d and 2-d kernel density estimates are sent to graphics window. Plot
  for 3-d is generated by the \code{misc3d} and \code{rgl}
  libraries and is sent to RGL window.
}


\details{
  For \code{kde} objects, the function headers for the different dimensional data are 
  \preformatted{
  ## univariate
  plot(x, xlab, ylab="Density function", add=FALSE,
  drawpoints=FALSE, ptcol="blue", col="black", jitter=TRUE, ...)

  ## bivariate
  plot(x, display="slice", cont=c(25,50,75), abs.cont, 
    approx.cont=FALSE, xlab, ylab, zlab="Density function", add=FALSE, 
    drawpoints=FALSE, drawlabels=TRUE, theta=-30, phi=40, d=4,
    ptcol="blue", col="black", ...)

  ## trivariate
  plot(x, cont=c(25,50,75), abs.cont, approx.cont=FALSE, colors,
    add=FALSE, drawpoints=FALSE, alpha, alphavec, xlab, ylab, zlab, 
    size=3, ptcol="blue", ...)   
  }
  For \code{kda.kde} objects, the function headers for the different dimensional data are 
  \preformatted{
  ## univariate
  plot(x, y, y.group, prior.prob=NULL, xlim, ylim,
    xlab="x", ylab="Weighted density function", drawpoints=FALSE,
    col, ptcol, jitter=TRUE, ...)

  ## bivariate
  plot(x, y, y.group, prior.prob=NULL, cont=c(25,50,75),
    abs.cont, approx.cont=FALSE, xlim, ylim, xlab, ylab, drawpoints=FALSE,
    drawlabels=TRUE, col, partcol, ptcol, ...)

  ## trivariate
  plot(x, y, y.group, prior.prob=NULL, cont=c(25,50,75),
    abs.cont,  approx.cont=FALSE, colors, alphavec, xlab, ylab, zlab, 
    drawpoints=FALSE, size=3, ptcol="blue", ...)
  }
 
  The arguments are
  \describe{
  \item{\code{display}}{type of display, "slice" for contour plot, "persp" for perspective plot, 
    "image" for image plot, "filled.contour" for filled contour plot (1st form),
   "filled.contour2" (2nd form) (2-d plot)}
  \item{\code{cont}}{vector of percentages for contour level curves}
  \item{\code{abs.cont} }{ vector of absolute density estimate heights for contour level curves}
  \item{\code{approx.cont} }{ flag to compute approximate contour levels. Default is FALSE.} 
  \item{\code{ptcol} }{ plotting colour for data points}
  \item{\code{col} }{ plotting colour for density estimate (1-d, 2-d plot)}
  \item{\code{colors} }{ vector of colours for each contour (3-d plot)}
  \item{\code{jitter} }{ flag to jitter rug plot (1-d plot). Default is TRUE.}
  \item{\code{xlim, ylim} }{ axes limits}
  \item{\code{xlab,ylab,zlab} }{ axes labels}
  \item{\code{add} }{ flag to add to current plot. Default is FALSE.}
  \item{\code{theta,phi,d} }{ graphics parameters for perspective plots (2-d plot)}
  \item{\code{drawpoints} }{ flag to draw data points on density estimate. Default is FALSE.}
  \item{\code{drawlabels} }{ flag to draw contour labels (2-d plot). Default is TRUE.}
  \item{\code{alpha} }{ transparency value of plotting symbol (3-d plot)}
  \item{\code{alphavec} }{ vector of transparency values for contours (3-d plot)}
  \item{\code{size} }{ size of plotting symbol (3-d plot)}
  \item{\code{prior.prob} }{ vector of prior probabilities}  
  \item{\code{partcol} }{ vector of colours for partition classes (1-d, 2-d plot)} 
  }

  The 1-d plot is a standard plot of a 1-d curve. If
  \code{drawpoints=TRUE} then a rug plot is added.
  
  There are different types of plotting displays for 2-d data available,
  controlled by the \code{display} parameter.  (a) If \code{display="slice"} then a slice/contour plot
  is generated using \code{contour}.  
  (b) If \code{display} is \code{"filled.contour"} or \code{"filled.contour2"} 
  then a filled contour plot is generated.
  The default contours are at 25\%, 50\%, 75\% or
  \code{cont=c(25,50,75)} which are upper percentages of
  highest density regions. See below for alternative
  contour levels. (c) \code{display="persp"} then a perspective/wire-frame plot
  is generated.  The default z-axis limits \code{zlim} are the default
  from the usual \code{persp} command. (d) If \code{display="image"} then an image plot
  is generated. Default colours are the default from the usual
  \code{image} command.

  For 3-dimensional data, the interactive plot is a series of nested
  3-d contours. 
  The default contours are \code{cont=c(25,50,75)}. See below for
  alternative  contour levels. The
  default \code{colors} are \code{heat.colors} and the
  default opacity \code{alphavec} ranges from 0.1 to 0.5.

  To specify contours, either one of \code{cont} or \code{abs.cont}
  is required. \code{cont} specifies upper percentages which
  correspond to probability contour regions. If \code{abs.cont} is set to particular
  values, then contours at these levels are drawn.
  This second option is useful for plotting
  multiple density estimates with common contour levels. See
  \code{\link{contourLevels}} for details on computing contour levels.   
  If \code{approx=FALSE}, then the exact KDE at x is computed. Otherwise the exact 
  KDE is replaced by the KDE at the nearest 
  grid point. This can dramatically reduce computation time for large data sets.
}
 

\seealso{\code{\link{kde}}, \code{\link{kda.kde}}}

\examples{
## kde examples

## univariate example
x <- rnorm.mixt(n=100, mus=1, sigmas=1, props=1)
fhat <- kde(x=x, h=hpi(x))  
plot(fhat)

## bivariate example
data(unicef)
H.scv <- Hscv(x=unicef)
fhat <- kde(x=unicef, H=H.scv, compute.cont=TRUE)
plot(fhat, drawpoints=TRUE, drawlabels=FALSE, col=3, lwd=3, cex=0.1)
plot(fhat, display="persp", border=NA, col="grey96", shade=0.75)
plot(fhat, display="image", col=rev(heat.colors(100)))
plot(fhat, display="filled.contour2", cont=seq(10,90,by=10))

## pair of densities with same absolute contour levels
x <- rmvnorm.mixt(n=100, mus=c(0,0), Sigmas=diag(2), props=1)
Hx <- Hpi(x)
fhatx <- kde(x=x, H=Hx, xmin=c(-4,-4), xmax=c(4,4)) 
y <- rmvnorm.mixt(n=100, mus=c(0.5,0.5), Sigmas=0.5*diag(2), props=1)
Hy <- Hpi(y)
fhaty <- kde(x=y, H=Hy, xmin=c(-4,-4), xmax=c(4,4))
lev <- contourLevels(fhatx, prob=c(0.25, 0.5, 0.75))
plot(fhatx, abs.cont=lev)
plot(fhaty, abs.cont=lev, col=3, add=TRUE) 

## large sample from bivariate normal 
x <- rmvnorm.mixt(5000, c(0,0), invvech(c(1, 0.8, 1)))    
H <- Hpi(x, binned=TRUE)
fhat <- kde(x, H=H, compute.cont=TRUE, approx.cont=TRUE)
plotmixt(mus=c(0,0), Sigmas=invvech(c(1, 0.8, 1)), props=1)      
plot(fhat, drawlabels=FALSE, add=TRUE, col=2)  

## trivariate example
library(MASS)
x <- iris[,1:3]
H.pi <- Hpi(x, pilot="samse")
fhat <- kde(x, H=H.pi, compute.cont=TRUE)  
plot(fhat, drawpoints=TRUE)

## kda.kde examples

## univariate example
ir <- iris[,1]
ir.gr <- iris[,5]
hs <- hkda(x=ir, x.gr=ir.gr)
kda.fhat <- kda.kde(ir, ir.gr, hs=hs, xmin=3, xmax=9)
plot(kda.fhat, xlab="Sepal length")

## bivariate example
ir <- iris[,1:2]
ir.gr <- iris[,5]
H <- Hkda(ir, ir.gr, bw="plugin", pre="scale")
kda.fhat <- kda.kde(ir, ir.gr, Hs=H)
plot(kda.fhat, abs.cont=0)

## trivariate example
## colour indicates species, transparency indicates density heights
ir <- iris[,1:3]
ir.gr <- iris[,5] 
H <- Hkda(ir, ir.gr, bw="plugin")
kda.fhat <- kda.kde(ir, ir.gr, Hs=H, compute.cont=TRUE)
plot(kda.fhat)   
}


\keyword{hplot}
