% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/landform.R
\name{landform}
\alias{landform}
\title{Landform Classification}
\usage{
landform(
  x,
  scale = 3,
  sn = 3,
  ln = 7,
  class.type = "slope.position",
  descriptive = FALSE,
  leg.pos = "topright",
  leg.cex = 0.65
)
}
\arguments{
\item{x}{A RasterLayer or a SpatRaster object representing the landscape.}

\item{scale}{The size of the neighbourhood for calculating the TPI. Default is 3.}

\item{sn}{The size of the small neighbourhood for calculating the TPI in the Landform Classification. Default is 3.}

\item{ln}{The size of the large neighbourhood for calculating the TPI in the Landform Classification. Default is 7.}

\item{class.type}{The type of classification to be performed. Options are "slope.position" for Slope Position Classification
and "landform.classification" for Landform Classification. Default is "slope.position".}

\item{descriptive}{A logical parameter (default is FALSE). If set to TRUE, the function will calculate and return
additional descriptive statistics for each class. These statistics include the count of pixels, total area (in Km2), and percentage
of the total area for each class. Additionally, a bar plot showing the area of each class will be generated.}

\item{leg.pos}{The position of the legend in the plot. Default is "topright".}

\item{leg.cex}{The magnification to be used for sizing the legend relative to the current setting of 'cex'. Default is 0.65.}
}
\value{
The package plots the raster of all the classes combined. If the \code{descriptive} parameter is set to \code{TRUE},
a bar plot showing the area of each class is generated. A list of SpatRaster objects
representing the different classes of the landscape is also returned. The list will contain an item named
\code{Descriptive Statistics} if the \code{descriptive} parameter is set to \code{TRUE}. For each class, the item reports the
cell counts, the corresponding area (in Km2), and the corresponding percentage.\cr

For the "slope.position" classification, the returned layers are:
\itemize{
  \item "all": all classes combined
  \item "Valley": TPI <= -1
  \item "Lower Slope": -1 < TPI <= -0.5
  \item "Flat Slope": -0.5 < TPI < 0.5 and slope <= 5
  \item "Middle Slope": -0.5 < TPI < 0.5 and slope > 5
  \item "Upper Slope": 0.5 < TPI <= 1
  \item "Ridge": TPI > 1
}

For the "landform.classification", the returned layers are:
\itemize{
  \item "all": all classes combined
  \item "Canyons-Deeply Incised Streams": sn <= -1 and ln <= -1
  \item "Midslope Drainage-Shallow Valleys": sn <= -1 and -1 < ln < 1
  \item "Upland Drainages-Headwaters": sn <= -1 and ln >= 1
  \item "U-shaped Valleys": -1 < sn < 1 and ln <= -1
  \item "Plains": -1 < sn < 1 and -1 < ln < 1 and slope <= 5
  \item "Open Slopes": -1 < sn < 1 and -1 < ln < 1 and slope > 5
  \item "Upper Slopes-Mesas": -1 < sn < 1 and ln >= 1
  \item "Local Ridges-Hills in Valley": sn >= 1 and ln <= -1
  \item "Midslopes Ridges-Small Hills in Plains": sn >= 1 and -1 < ln < 1
  \item "Mountain Tops-High Ridges": sn >= 1 and ln >= 1
}
}
\description{
This function classifies a landscape into different categories based on the Topographic Position Index (TPI) and slope.
It offers two types of classifications: Slope Position Classification (Weiss 2001) and Landform Classification (Weiss 2001; Jenness 2003).\cr
Visit this \href{https://drive.google.com/file/d/1vjy8_HPtuyKkzSkYwKdQbHJLrVtO79ft/view?usp=sharing}{LINK} to access the package's vignette.\cr
}
\details{
The function internally calculates the Standardised Topographic Position Index (TPI) for the given landscape.
The TPI is calculated as the difference between the elevation of each cell in the landscape and the mean elevation of
its neighbouring cells. The TPI is then standardised by dividing the difference between each cell's TPI and the mean TPI
of its neighbouring cells by the standard deviation of the TPI values within the neighbourhood.

The Slope Position Classification uses the standardised TPI and slope to classify the landscape into
six categories: Valley, Lower Slope, Flat Slope, Middle Slope, Upper Slope, and Ridge.
This classification is useful for identifying the position of a location on a slope (Weiss 2001).

The Landform Classification uses two standardised TPI grids at different scales to classify the landscape
into ten categories: Canyons, Midslope Drainage, Upland Drainage, U-shaped Valleys, Plains, Open Slopes,
Upper Slopes, Local Ridges, Midslope Ridges, and Mountain Tops. This classification is useful for identifying
broader landform types (Weiss 2001; Jenness 2003).

As for the descriptive statistics returned by the function, please note that if the input raster layer is not in
a projected coordinate system, the area calculation for each class (which is based on the resolution of the raster and the
count of cells in each class) may not accurately represent the true area on the ground. Therefore, it's recommended to use
a raster layer in a suitable projected coordinate system for your area of interest to ensure accurate area calculations.
}
\examples{
# EXAMPLE 1

# Get an elevation dataset to be used
require("spatialEco")
elev <-terra::rast(system.file("extdata/elev.tif", package="spatialEco"))

# Run the analysis
result <- landform(elev, scale = 3, class.type = "slope.position")

# EXAMPLE 2

# Get an elevation dataset to be used
require("spatialEco")
elev <- terra::rast(system.file("extdata/elev.tif", package="spatialEco"))

# Run the analysis
result <- landform(elev, class.type = "landform.classification", sn=3, ln=11)

}
\references{
Weiss, A. (2001). Topographic Position and Landforms Analysis. Poster presentation, ESRI User Conference, San Diego, CA.

Jenness, J. (2003). TPI ArcView Extension. Jenness Enterprises. Available at: http://www.jennessent.com/arcview/tpi.htm
}
\seealso{
\code{\link[raster]{raster}}, \code{\link[terra]{rast}}, \code{\link[terra]{focal}}, \code{\link[terra]{terrain}}, \code{\link[terra]{plot}}
}
