% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lemna.R
\name{lemna}
\alias{lemna}
\alias{lemna.default}
\alias{lemna.lemna_scenario}
\title{Simulate a Lemna scenario}
\usage{
lemna(...)

\method{lemna}{default}(
  init = c(BM = 0, M_int = 0),
  times,
  param,
  envir,
  ode_mode = c("r", "c"),
  nout = 2,
  ...
)

\method{lemna}{lemna_scenario}(x, init, times, param, envir, ...)
}
\arguments{
\item{...}{additional parameters passed on to \code{\link[deSolve:ode]{deSolve::ode()}}}

\item{init}{initial state of the model variables}

\item{times}{numeric vector, output times for which model results are returned}

\item{param}{named list, Lemna model parameters}

\item{envir}{named list, contains time series data for each of the five
environmental variables}

\item{ode_mode}{character, switch controls if ODE functions implemented in R or
C are used, defaults to R}

\item{nout}{numeric, controls the number of additional output variables of the
model, such as \code{C_int} (internal concentration) or \code{FrondNo} (the number
of fronds), see e.g. \code{\link[deSolve:lsoda]{deSolve::lsoda()}} for details. Defaults to \code{2}}

\item{x}{a \code{lemna_scenario} object}
}
\value{
\code{data.frame} with simulation results
}
\description{
The function numerically integrates the Lemna model using the supplied
parameters, environmental factor time series, output time points, and other
settings. Numerical integration is handled by \code{\link[deSolve:lsoda]{deSolve::lsoda()}} by default
which is well suited to handle stiff and non-stiff numerical problems.
}
\details{
\subsection{Initial state}{

The model has two state variables:
\itemize{
\item Biomass, \code{BM} (g dw m-2)
\item Mass of toxicant in plant population, \code{M_int} (mass per m2, e.g. ug m-2)
}
}

\subsection{Output times}{

The function will only return results for the requested time points. The solver
may make additional time steps which are not returned, depending on the
solver settings such as numerical tolerance. The number of output times and
the distance between them can have influence on the numerical precision of
results. Please refer to the manual of the \link{deSolve} package for more
information on that topic.
}

\subsection{Parameters}{

The list of available model parameters, their units, and suggested default
values is documented in \code{\link[=param_defaults]{param_defaults()}}.
}

\subsection{Environmental variables}{

The model requires a time series for each of the five environmental variables
For ease of use, a time series can be represented by either a constant
numeric value, a \code{data.frame} containing a time series, a character
string with a path to a CSV file containing a time series, or a function
returning a value for a certain time point.

The five environmental variables are as follows:
\itemize{
\item Exposure concentration, \code{conc} (mass per volume, e.g ug L-1)
\item Temperature, \code{tmp} (°C)
\item Irradiance, \code{irr} (kJ m-2 d-1)
\item Phosphorus concentration, \code{P} (mg P L-1)
\item Nitrogen concentration, \code{N} (mg N L-1)
}

A \code{data.frame} containing a time series must consist of exactly two columns:
The first column contains the time in days, the second column the environmental
factor in question. The \code{data.frame} must contain at least one row. Time
points are not required to coincide with the solver's output times but
the user should take care that the solver's time step length is sufficiently
small so that it can consider the relevant dynamics of the time series.
This problem can be avoided by calling the model ODEs implemented in C, see
argument \code{ode_mode}.

If a string is passed as an environmental factor, the function will interpret
the string as a file path and will try to load the contents of a CSV file.
The same requirements as for a \code{data.frame} apply to a time series loaded
from a file.

When using the pure R code ODE, it is also possible to pass a function as
an environmental factor. The function must accept exactly one argument which
represents time and must return a single scalar value. Using a function
in combination with the compiled code solver will raise an error.
}

\subsection{R vs. compiled code}{

The model can be simulated using pure \emph{R} code (the default) or an implementation
that uses the compiled code feature of \link{deSolve}. Compiled code is almost
always significantly faster than pure \emph{R}. The solver for compiled ODEs also
handles environmental variables better than the pure \emph{R} version. For optimal
performance, the time series of environmental variables should always be as
short as possible and as long as needed.

To use the compiled code feature, set the argument \code{ode_mode = "c"}.
}

\subsection{Additional outputs}{

For reasons of convenience, the return value contains by default two additional
variables derived from simulation results: the internal concentration \code{C_int}
as well as the number of fronds \code{FrondNo}. These can be disabled by setting
the argument \code{nout = 0}.

The compiled code model can return up to 18 additional variables which
represent intermediary variables, environmental variables, and derivatives
calculated within the model equations. Please refer to the model description
of \emph{Klein et al.} for more information on these variables and how they
influence model behavior.
}
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: All scenario parameters supplied as arguments

\item \code{lemna_scenario}: Scenario parameters supplied as a \code{lemna_scenario} object
}}

\examples{
# Simulate the metsulfuron example scenario
lemna(metsulfuron)

# Create a simple plot of the number of fronds
lemna(metsulfuron) -> result
plot(result$time, result$FrondNo)

# Create a nicer plot using a dedicated plotting routine
plot(result)

# Simulate the example scenario for a period of 30 days
lemna(metsulfuron, times=0:30) -> result30
plot(result30)

##
## Create a custom Lemna scenario from scratch
##

# Initial state: 12 fronds, no toxicant mass
myinit <- c(BM=0.0012, M_int=0)

# Output times: simulate 7 days with a ~2 hour time step
mytime <- seq(0, 7, 0.1)

# Default model parameters + TKTD parameters of a hypothetical substance
myparam <- param_defaults(list(
  EC50_int = 0.1, # 50\% effect level (ug L-1)
  b = 0.7,        # slope parameter (-)
  P = 0.01        # permeability (cm d-1)
))

# Custom environmental variables
myenvir <- list(
  # exposure step function:
  # 3 days no exposure, followed by 4 days of 10 ug L-1
  conc = data.frame(t=c(0,3,4,7), conc=c(0,0,10,10)),
  tmp = 18,    # constant temperature of 18 °C
  irr = 15000, # constant irradiance of 15,000 kJ m-2 d-1
  N = 0.6,     # constant Nitrogen concentration of 0.6 mg L-1
  P = 0.3      # constant Phosphorus concentration of 0.3 mg L-1
)

# Simulate the custom scenario and plot results
lemna(init=myinit, times=mytime, param=myparam, envir=myenvir) -> result_custom
plot(result_custom)

# Simulate again, forcing the solver to use smaller time steps of hmax=0.001.
# The resulting curves are almost identical for this example.
lemna(init=myinit, times=mytime, param=myparam, envir=myenvir, hmax=0.001) -> result_custom2
library(ggplot2)
ggplot(result_custom, aes(time, FrondNo)) +
  geom_line() +
  geom_line(data=result_custom2, color="red", style="dashed")

# Combine all settings into a scenario object and simulate it
scenario <- new_lemna_scenario(
 init = myinit,
 param = myparam,
 times = mytime,
 envir = myenvir
)
lemna(scenario)
}
