% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\name{predict.lgspline}
\alias{predict.lgspline}
\title{Predict Method for Fitted Lagrangian Multiplier Smoothing Spline}
\usage{
\method{predict}{lgspline}(
  object,
  newdata = NULL,
  parallel = FALSE,
  cl = NULL,
  chunk_size = NULL,
  num_chunks = NULL,
  rem_chunks = NULL,
  B_predict = NULL,
  take_first_derivatives = FALSE,
  take_second_derivatives = FALSE,
  expansions_only = FALSE,
  new_predictors = NULL,
  ...
)
}
\arguments{
\item{object}{A fitted lgspline model object containing model parameters and fit}

\item{newdata}{Matrix or data.frame; New predictor values for out-of-sample prediction.
If NULL (default), uses training data}

\item{parallel}{Logical; whether to use parallel processing for prediction computations.
Experimental feature - use with caution. Default FALSE}

\item{cl}{Optional cluster object for parallel processing. Required if parallel=TRUE.
Default NULL}

\item{chunk_size}{Integer; Size of computational chunks for parallel processing.
Default NULL}

\item{num_chunks}{Integer; Number of chunks for parallel processing. Default NULL}

\item{rem_chunks}{Integer; Number of remainder chunks for parallel processing.
Default NULL}

\item{B_predict}{Matrix; Optional custom coefficient matrix for prediction.
Default NULL (uses object$B internally).}

\item{take_first_derivatives}{Logical; whether to compute first derivatives of the
fitted function. Default FALSE}

\item{take_second_derivatives}{Logical; whether to compute second derivatives of the
fitted function. Default FALSE}

\item{expansions_only}{Logical; whether to return only basis expansions without
computing predictions. Default FALSE}

\item{new_predictors}{Matrix or data frame; overrides 'newdata' if provided.}

\item{...}{Additional arguments passed to internal prediction methods.}
}
\value{
Depending on the options selected, returns the following:
\describe{
  \item{predictions}{Numeric vector of predicted values (default case, or if derivatives requested).}
  \item{first_deriv}{Numeric vector of first derivatives (if take_first_derivatives = TRUE).}
  \item{second_deriv}{Numeric vector of second derivatives (if take_second_derivatives = TRUE).}
  \item{expansions}{List of basis expansions (if expansions_only = TRUE).}
}

With derivatives included, output is in the form of a list with elements
"preds", "first_deriv", and "second_deriv" for the vector of predictions,
first derivatives, and second derivatives respectively.
}
\description{
Generates predictions, derivatives, and basis expansions from a fitted lgspline model.
Supports both in-sample and out-of-sample prediction with optional parallel processing.
(Wrapper for internal predict method)
}
\details{
Implements multiple prediction capabilities:
\itemize{
  \item Standard prediction: Returns fitted values for new data points
  \item Derivative computation: Calculates first and/or second derivatives
  \item Basis expansion: Returns design matrix of basis functions
  \item Correlation structures: Supports non-Gaussian GLM correlation via
        variance-covariance matrices
}

If newdata and new_predictor are left NULL, default input used for model fitting
will be used. Priority will be awarded to new_predictor over newdata when
both are not NULL.

To obtain fitted values, users may also call model_fit$predict() or
model_fit$ytilde for an lgspline object "model_fit".

The parallel processing feature is experimental and should be used with caution.
When enabled, computations are split across chunks and processed in parallel,
which may improve performance for large datasets.
}
\examples{

## Generate example data
set.seed(1234)
t <- runif(1000, -10, 10)
y <- 2*sin(t) + -0.06*t^2 + rnorm(length(t))

## Fit model
model_fit <- lgspline(t, y)

## Generate predictions for new data
newdata <- matrix(sort(rnorm(10000)), ncol = 1) # Ensure matrix format
preds <- predict(model_fit, newdata)

## Compute derivative
deriv1_res <- predict(model_fit, newdata,
                      take_first_derivatives = TRUE)
deriv2_res <- predict(model_fit, newdata,
                      take_second_derivatives = TRUE)

## Visualize results
oldpar <- par(no.readonly = TRUE) # Save current par settings
layout(matrix(c(1,1,2,2,3,3), byrow = TRUE, ncol = 2))

## Plot function
plot(newdata[,1], preds,
       main = 'Fitted Function',
       xlab = 't',
       ylab = "f(t)", type = 'l')

## Plot first derivative
plot(newdata[,1],
       deriv1_res$first_deriv,
       main = 'First Derivative',
       xlab = 't',
       ylab = "f'(t)", type = 'l')

## Plot second derivative
plot(newdata[,1],
       deriv2_res$second_deriv,
       main = 'Second Derivative',
       xlab = 't',
       ylab = "f''(t)", type = 'l')

par(oldpar) # Reset to original par settings


}
\seealso{
\code{\link{lgspline}} for model fitting,
\code{\link{plot.lgspline}} for visualizing predictions
}
