% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\name{plot.lgspline}
\alias{plot.lgspline}
\title{Plot Method for Lagrangian Multiplier Smoothing Spline Models}
\usage{
\method{plot}{lgspline}(
  x,
  show_formulas = FALSE,
  digits = 4,
  legend_pos = "topright",
  custom_response_lab = "y",
  custom_predictor_lab = NULL,
  custom_predictor_lab1 = NULL,
  custom_predictor_lab2 = NULL,
  custom_formula_lab = NULL,
  custom_title = "Fitted Function",
  text_size_formula = NULL,
  legend_args = list(),
  new_predictors = NULL,
  xlim = NULL,
  ylim = NULL,
  color_function = NULL,
  add = FALSE,
  vars = c(),
  ...
)
}
\arguments{
\item{x}{A fitted lgspline model object containing the model fit to be plotted}

\item{show_formulas}{Logical; whether to display analytical formulas for each partition.
Default FALSE}

\item{digits}{Integer; Number of decimal places for coefficient display in formulas.
Default 4}

\item{legend_pos}{Character; Position of legend for 1D plots ("top", "bottom", "left",
"right", "topleft", etc.). Default "topright"}

\item{custom_response_lab}{Character; Label for response variable axis. Default "y"}

\item{custom_predictor_lab}{Character; Label for predictor axis in 1D plots. If NULL
(default), uses predictor column name}

\item{custom_predictor_lab1}{Character; Label for first predictor axis (x1) in 2D plots.
If NULL (default), uses first predictor column name}

\item{custom_predictor_lab2}{Character; Label for second predictor axis (x2) in 2D plots.
If NULL (default), uses second predictor column name}

\item{custom_formula_lab}{Character; Label for fitted response on link function scale.
If NULL (default), uses "link(E[custom_response_lab])" for non-Gaussian models
with non-identity link, otherwise uses custom_response_lab}

\item{custom_title}{Character; Main plot title. Default "Fitted Function"}

\item{text_size_formula}{Numeric; Text size for formula display. Passed to cex in legend()
for 1D plots and hover font size for 2D plots. If NULL (default), uses 0.8 for 1D
and 8 for 2D}

\item{legend_args}{List; Additional arguments passed to legend() for 1D plots}

\item{new_predictors}{Matrix; Optional new predictor values for prediction. If NULL
(default), uses original fitting data}

\item{xlim}{Numeric vector; Optional x-axis limits for 1D plots. Default NULL}

\item{ylim}{Numeric vector; Optional y-axis limits for 1D plots. Default NULL}

\item{color_function}{Function; Returns colors for plotting by partition, must return K+1 vector of valid colors. Defaults to NULL, in which case \code{grDevices::rainbow(K+1)} is used for 1D and \code{grDevices::colorRampPalette(RColorBrewer::brewer.pal(8, "Spectral"))(K+1)} used in multiple.}

\item{add}{Logical; If TRUE, adds to existing plot (1D only). Similar to add in
\code{\link[graphics]{hist}}. Default FALSE}

\item{vars}{Numeric or character vector; Optional indices for selecting variables to plot. Can either be numeric (the column indices of "predictors" or "data") or character (the column names, if available from "predictors" or "data")}

\item{...}{Additional arguments passed to underlying plot functions:
\itemize{
  \item 1D: Passed to \code{\link[graphics]{plot}}
  \item 2D: Passed to \code{\link[plotly]{plot_ly}}
}}
}
\value{
Returns
\describe{
  \item{1D}{Invisibly returns NULL (base R plot is drawn to device).}
  \item{2D}{Plotly object showing interactive surface plot.}
}
}
\description{
Creates visualizations of fitted spline models, supporting both 1D line plots and 2D surface
plots with optional formula annotations and customizable aesthetics.
(Wrapper for internal plot method)
}
\details{
Produces different visualizations based on model dimensionality:
\itemize{
  \item 1D models: Line plot showing fitted function across partitions, with optional
        data points and formula annotations
  \item 2D models: Interactive 3D surface plot using plotly, with hover text showing
        predicted values and optional formula display
}

Partition boundaries are indicated by color changes in both 1D and 2D plots.

When plotting using "select_vars" option, it is recommended to use the
"new_predictors" argument to set all terms not involved with plotting to 0
to avoid non-sensical results. But for some cases, it may be useful to set
other predictors fixed at certain values. By default, observed values in the
data set are used.

The function relies on linear expansions being present - if (for example) a
user includes the argument "_1_" or "_2_" in "exclude_these_expansions", then
this function will not be able to extract the predictors needed for plotting.

For this case, try constraining the effects of these terms to 0 instead using
"constraint_vectors" and "constraint_values" argument, so they are kept in
the expansions but their corresponding coefficients will be 0.
}
\examples{

## Generate example data
set.seed(1234)
t_data <- runif(1000, -10, 10)
y_data <- 2*sin(t_data) + -0.06*t_data^2 + rnorm(length(t_data))

## Fit model with 10 partitions
model_fit <- lgspline(t_data, y_data, K = 9)

## Basic plot
plot(model_fit)

## Customized plot with formulas
plot(model_fit,
     show_formulas = TRUE,         # Show partition formulas
     custom_response_lab = 'Price',  # Custom axis labels
     custom_predictor_lab = 'Size',
     custom_title = 'Price vs Size', # Custom title
     digits = 2,                    # Round coefficients
     legend_pos = 'bottom',         # Move legend
     text_size_formula = 0.375,     # Adjust formula text size
     pch = 16,                      # Point style
     cex.main = 1.25)               # Title size

}
\seealso{
\code{\link{lgspline}} for model fitting,
\code{\link[graphics]{plot}} for additional 1D plot parameters,
\code{\link[plotly]{plot_ly}} for additional 2D plot parameters
}
