% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lasground.r
\name{lasground}
\alias{lasground}
\title{Classify points as 'ground' or 'not ground'}
\usage{
lasground(las, algorithm, last_returns = TRUE)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{algorithm}{a ground-segmentation function. \code{lidR} has: \link{pmf} and \link{csf}.}

\item{last_returns}{logical. The algorithm will use only the last returns (including the first returns
in cases of a single return) to run the algorithm. If FALSE all the returns are used. If the attribute
\code{'ReturnNumber'} or \code{'NumberOfReturns'} are absent, \code{'last_returns'} is turned
to \code{FALSE} automatically.}
}
\value{
If the input is a \code{LAS} object, return a \code{LAS} object. If the input is a
\code{LAScatalog}, returns a \code{LAScatalog}.
}
\description{
Classify points as 'ground' or 'not ground' with several possible algorithms. The function updates the
attribute \code{Classification} of the LAS object. The points classified as 'ground' are assigned
a value of 2 according to \href{http://www.asprs.org/a/society/committees/standards/LAS_1_4_r13.pdf}{las specifications}.
}
\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} (in bold). For more details see the
\link[lidR:LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item \strong{chunk size}: How much data is loaded at once.
\item \strong{chunk buffer*}: Mandatory to get a continuous output without edge effects. The buffer
is always removed once processed and will never be returned either in R or in files.
\item \strong{chunk alignment}: Align the processed chunks.
\item \strong{progress}: Displays a progression estimation.
\item \strong{output files*}: Mandatory because the output is likely to be too big to be returned
in R and needs to be written in las/laz files. Supported templates are \code{\{XLEFT\}}, \code{\{XRIGHT\}},
\code{\{YBOTTOM\}}, \code{\{YTOP\}}, \code{\{XCENTER\}}, \code{\{YCENTER\}} \code{\{ID\}} and, if
chunk size is equal to 0 (processing by file), \code{\{ORIGINALFILENAME\}}.
\item select: The function will write files equivalent to the original ones. Thus \code{select = "*"}
and cannot be changed.
\item \strong{filter}: Read only points of interest.
}
}

\examples{
LASfile <- system.file("extdata", "Topography.laz", package="lidR")
las <- readLAS(LASfile, select = "xyzrn")

# Using the Progressive Morphological Filter
# --------------------------------------

ws  <- seq(3,12, 3)
th  <- seq(0.1, 1.5, length.out = length(ws))

las <- lasground(las, pmf(ws, th))
plot(las, color = "Classification")

#' # Using the Cloth Simulation Filter
# --------------------------------------

# (Parameters chosen mainly for speed)
mycsf <- csf(TRUE, 1, 1, time_step = 1)
las <- lasground(las, mycsf)
plot(las, color = "Classification")
}
