% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gap.detection.R
\name{gapDetection}
\alias{gapDetection}
\title{Gap detection in a Canopy Height Model}
\usage{
gapDetection(
  chm,
  ratio = 2,
  gap.max.height = 1,
  min.gap.surface = 25,
  max.gap.surface = +Inf,
  closing.height.bin = 1,
  nlFilter = "Median",
  nlsize = 3,
  gapReconstruct = FALSE
)
}
\arguments{
\item{chm}{raster object. canopy height model}

\item{ratio}{numeric. maximum ratio between surrounding canopy height and gap distance (a pixel belongs to the gap only if for any vegetation pixel around it, the distance to the vegetation pixel is larger than pixel height/ratio). If ratio is set to NULL, this criterion is not taken into account}

\item{gap.max.height}{numeric. maximum canopy height to be considered as gap}

\item{min.gap.surface}{numeric. minimum gap surface}

\item{max.gap.surface}{numeric. maximum gap surface}

\item{closing.height.bin}{numeric. height bin width for morphological closing of gaps to test ratio between canopy height and gap distance}

\item{nlFilter}{string. type of non-linear filter to apply to canopy height model to remove artefacts, should be an option of \code{\link{demFiltering}}}

\item{nlsize}{numeric. kernel width in pixel for non-linear filtering}

\item{gapReconstruct}{boolean. default behaviour is that areas that do not fulfill the ratio criterion are removed from gaps. If set to TRUE, in case some pixels of a gap fulfill the distance criterion, the connected pixels that fulfill the height criterion are also integrated to it.}
}
\value{
A list of three raster objects: raster with gap labels, raster with gap surface, canopy height model after filter.
}
\description{
Performs gaps detection in a canopy height model. Function \code{\link{demFiltering}} is first applied to the canopy height model to remove artefacts. Gaps are then extracted based on several criteria:
\enumerate{
\item Vegetation height must be smaller than a threshold
\item Gap width must be large enough, depending on surrounding canopy height; distance to surrounding vegetation is tested with morphological closings
\item Gap must have a minimum surface
}
}
\examples{
data(chmchablais3)

# fill NA values in canopy height model
chmchablais3[is.na(chmchablais3)] <- 0

# gap detection with distance larger than canopy height / 2
gaps <- gapDetection(chmchablais3, ratio=2, gap.max.height=1, min.gap.surface=0)

# gap detection with distance larger than canopy height / 2
# and reconstruction of border areas
gaps1 <- gapDetection(chmchablais3, ratio=2, gap.max.height=1, min.gap.surface=0,
gapReconstruct=TRUE)

# gap detection without distance criterion
gaps2 <- gapDetection(chmchablais3, ratio=NULL, gap.max.height=1, min.gap.surface=0)

# gap id and corresponding surface for third detection parameters
table(raster::values(gaps2$gap.id))*raster::res(gaps2$gap.id)[1]^2

# plot original image
raster::plot(chmchablais3, main="Initial image")

# plot binary image of gaps
raster::plot(gaps$gap.id>0, main="Gaps", legend=FALSE)
raster::plot(gaps1$gap.id>0, main="Gaps, with reconstruction", legend=FALSE)
raster::plot(gaps2$gap.id>0, main="Gaps, no width criterion", legend=FALSE)

# plot filtered CHM
raster::plot(gaps2$filled.chm, main="Filtered CHM")

}
\seealso{
\code{\link{demFiltering}}, \code{\link{edgeDetection}}
}
