% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comparisons.R
\name{comparisons}
\alias{comparisons}
\title{Experimental function to compute contrasts between adjusted predictions}
\usage{
comparisons(
  model,
  variables = NULL,
  newdata = NULL,
  type = "response",
  vcov = TRUE,
  contrast_factor = "reference",
  contrast_numeric = 1,
  ...
)
}
\arguments{
\item{model}{Model object}

\item{variables}{Variables to consider (character vector). \code{NULL}
calculates marginal effects for all terms in the model object.}

\item{newdata}{A dataset over which to compute marginal effects. \code{NULL} uses
the original data used to fit the model.}

\item{type}{Type(s) of prediction as string or character vector. This can
differ based on the model type, but will typically be a string such as:
"response", "link", "probs", or "zero".}

\item{vcov}{Matrix or boolean
\itemize{
\item FALSE: does not compute unit-level standard errors. This can speed up computation considerably.
\item TRUE: computes unit-level standard errors using the default \code{vcov(model)} variance-covariance matrix.
\item Named square matrix: computes standard errors with a user-supplied variance-covariance matrix. This matrix must be square and have dimensions equal to the number of coefficients in \code{get_coef(model)}.
}}

\item{contrast_factor}{string
\itemize{
\item "reference": Each factor level is compared to the factor reference (base) level
\item "sequential": Each factor level is compared to the previous factor level
\item "pairwise": Each factor level is compared to all other levels
}}

\item{contrast_numeric}{string or numeric
\itemize{
\item Numeric of length 1: Contrast between the observed value and the observed value plus \code{contrast_numeric}
\item Numeric vector of length 2: Contrast between the 2nd element and the 1st element of the \code{contrast_numeric} vector.
\item "iqr": Contrast across the interquartile range of the regressor.
\item "sd": Contrast across one standard deviation around the regressor mean.
\item "2sd": Contrast across two standard deviations around the regressor mean.
\item "minmax": Contrast between the maximum and the minimum values of the regressor.
}}

\item{...}{Additional arguments are passed to the \code{predict()} method used to
compute adjusted predictions. These arguments are particularly useful for
mixed-effects or bayesian models (see the online vignettes on the
\code{marginaleffects} website). Available arguments can vary from model to
model, depending on the range of supported arguments by each modeling
package. See the "Model-Specific Arguments" section of the
\code{?marginaleffects} document for a non-exhaustive list of available
arguments.}
}
\description{
This function calculates contrasts (or comparisons) between adjusted
predictions for each row of the dataset. The resulting object can processed
by the \code{tidy()} or \code{summary()} functions, which compute Average Contrasts.
The \code{datagrid()} function and the \code{newdata} argument can be used to
calculate contrasts Contrasts at the Mean or Contrasts at User-Specified
values (aka Contrasts at Representative values). Additional information can
be found in the Details and Examples sections below, and in the vignette on
the \code{marginaleffects} website.
}
\details{
A "contrast" is the difference between two adjusted predictions, calculated
for meaningfully different regressor values (e.g., College graduates vs.
Others). Uncertainty estimates are computed using the delta method.

Detailed vignettes on contrasts, marginal effects, predictions, and marginal
means, as well as a list of supported models can be found on the package
website:

https://vincentarelbundock.github.io/marginaleffects/
}
\examples{
library(marginaleffects)
library(magrittr)

# Linear model
tmp <- mtcars
tmp$am <- as.logical(tmp$am)
mod <- lm(mpg ~ am + factor(cyl), tmp)
comparisons(mod, contrast_factor = "reference") \%>\% tidy()
comparisons(mod, contrast_factor = "sequential") \%>\% tidy()
comparisons(mod, contrast_factor = "pairwise") \%>\% tidy()

# GLM with different scale types
mod <- glm(am ~ factor(gear), data = mtcars)
comparisons(mod) \%>\% tidy()
comparisons(mod, type = "link") \%>\% tidy()

# Numeric contrasts
mod <- lm(mpg ~ hp, data = mtcars)
comparisons(mod, contrast_numeric = 1) \%>\% tidy()
comparisons(mod, contrast_numeric = 5) \%>\% tidy()
comparisons(mod, contrast_numeric = c(90, 100)) \%>\% tidy()
comparisons(mod, contrast_numeric = "iqr") \%>\% tidy()
comparisons(mod, contrast_numeric = "sd") \%>\% tidy()
comparisons(mod, contrast_numeric = "minmax") \%>\% tidy()

}
