### SUMT function borrowed from 'clue' package
### 
### Adapted for linear constraints
sumt <- function(fn, grad=NULL, hess=NULL,
                 start,
                 maxRoutine, constraints, 
                 SUMTTol = sqrt(.Machine$double.eps),
                 SUMTQ = 10,
                 SUMTRho0 = NULL,
                 print.level=0,
                 SUMTMaxIter=100,
                 ...) {
   ## constraints    list w/components eqA and eqB.  Maximization will
   ##                be performed wrt to the constraint
   ##                A %*% theta + B = 0
   ##                The user must ensure the matrices are in correct
   ##                form
   ## maxSUMTiter    how many SUMT iterations to perform max
   ##
   penalty <- function(theta) {
      p <- A %*% theta + B
      sum(p*p)
   }
   ## Penalty gradient and Hessian are used only if corresponding function
   ## for the likelihood function is provided
   gPenalty <- function(theta) {
      2*(t(theta) %*% t(A) %*% A - t(B) %*% A)
   }
   hessPenalty <- function(theta) {
      2*t(A) %*% A
   }
   func <- function(theta, ...) {
      sum(fn(theta, ...))
   }
   funcS <- function(theta, ...) {
      ## this wrapper makes a) single-valued function (in case of BHHH
      ## vector-valued); and b) strips the 'maxRoutine' extra arguments
      f <- match.call()
      f[names(formals(maxRoutine))] <- NULL
      f[[1]] <- as.name("fn")
      names(f)[2] <- ""
      f1 <- eval(f, sys.frame(sys.parent()))
      sum(f1)
   }
   gradient <- function(theta, ...) {
      if(!is.null(grad)) {
         g <- grad(theta, ...)
         if(!is.null(dim(g))) {
            if(nrow(g) > 1) {
               g <- colSums( g )
            }
         }
         names( g ) <- names( start )
         return( g )
      }
      g <- numericGradient(func, theta, ...)
      if(!is.null(dim(g))) {
         return(colSums(g))
      } else {
         return(g)
      }
   }
   gradientS <- function(theta, ...) {
      g <- match.call()
      g[names(formals(maxRoutine))] <- NULL
      if(!is.null(grad)) {
         g[[1]] <- as.name("grad")
         names(g)[2] <- ""
         g <- eval(g, sys.frame(sys.parent()))
         if(!is.null(dim(g))) {
            if(nrow(g) > 1) {
               g <- colSums( g )
            }
         }
         names( g ) <- names( start )
         return( g )
      }
      g[[1]] <- as.name("numericGradient")
      names(g)[2] <- "t0"
      g$f <- func
      g <- eval(g, sys.frame(sys.parent()))
      if(!is.null(dim(g))) {
         return(colSums(g))
      } else {
         return(g)
      }
   }
   hessianS <- function(theta, ...) {
      ## just used for computing the final hessian, eventually using the
      ## supplied analytic information
      h <- match.call()
      h[names(formals(maxRoutine))] <- NULL
      if(!is.null(hess)) {
         h[[1]] <- as.name("hess")
         names(h)[2] <- ""
         h <- eval(h, sys.frame(sys.parent()))
      } else {
         h[[1]] <- as.name("numericHessian")
         names(h)[2] <- "t0"
         h$f <- func
         h$grad <- gradient
         h <- eval(h, sys.frame(sys.parent()))
      }
      rownames( h ) <- colnames( h ) <- names( start )
      return( h )
   }
   ## the penalized objective function
   Phi <- function(theta, ...) {
      funcS(theta, ...) - rho * penalty(theta)
   }
   if(!is.null(grad)) {
      gradPhi<- function(theta, ...) {
         g <- grad(theta, ...)
         if(is.matrix(g))
             g <- colSums(g)
         g - rho*gPenalty(theta)
      }
   }
   else
       gradPhi <- NULL
   if(!is.null(hess)) {
      hessPhi <- function(theta, ...) 
         hess(theta, ...) - rho*hessPenalty(theta)
   }
   else
       hessPhi <- NULL
   ##
   A <- constraints$eqA
   B <- constraints$eqB
    ## <NOTE>
    ## For the penalized minimization, the Newton-type nlm() may be
    ## computationally infeasible (although it works much faster for
    ## fitting ultrametrics to the Phonemes data).
    ## De Soete recommends using Conjugate Gradients.
    ## We provide a simple choice: by default, optim(method = "CG") is
    ## used.  If method is non-null and not "nlm", we use optim() with
    ## this method.  In both cases, control gives the control parameters
    ## for optim().
    ## If method is "nlm", nlm() is used, in which case control is
    ## ignored.  Note that we call nlm() with checking analyticals
    ## turned off, as in some cases (e.g. when fitting ultrametrics) the
    ## penalty function is not even continuous ...
   ## 
    ## Note also that currently we do not check whether optimization was
    ## "successful" ...
    ## </NOTE>
   ##
   rho <- 0
   result <- maxRoutine(fn=Phi, grad=gradPhi, hess=hessPhi,
                   start=start,
                   print.level=max(print.level - 1, 0),
                   ...)
   theta <- coef(result)
   if(print.level > 0) {
      cat("SUMT initial: rho = ", rho,
          ", function = ", funcS(theta, ...),
          ", penalty = ", penalty(theta), "\n")
      cat("Estimate:")
      print(theta)
   }
   ## <TODO>
   ## Better upper/lower bounds for rho?
   if(is.null(SUMTRho0))
       rho <- max(funcS(start, ...), 1e-3)/max(penalty(start), 1e-3)
   else
       rho <- SUMTRho0
   ## </TODO>
   iter <- 1L
   repeat {
      ## <TODO>
      ## Shouldnt't we also have maxiter, just in case ...?
      ## </TODO>
      thetaOld <- theta
      result <- maxRoutine(fn=Phi, grad=gradPhi, hess=hessPhi,
                      start=thetaOld,
                      print.level=max(print.level - 1, 0),
                      ...)
      theta <- coef(result)
      if(print.level > 0) {
         cat("SUMT iteration ", iter,
             ": rho = ", rho, ", function = ", funcS(theta, ...),
             ", penalty = ", penalty(theta), "\n", sep="")
         cat("Estimate:")
         print(theta)
      }
      if(max(abs(thetaOld - theta)) < SUMTTol)
          break
      if(iter >= SUMTMaxIter)
          break
      iter <- iter + 1L
      rho <- SUMTQ * rho
   }
   ## Now we replace the resulting gradient and Hessian with those,
   ## calculated on the original function
   result$gradient <- gradientS(theta, ...)
   result$hessian <- hessianS(theta, ...)
   result$constraints <- list(type="SUMT",
                             barrier.value=penalty(theta),
                             outer.iterations=iter
                             )
   return(result)
}
