% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simca.R
\name{simca}
\alias{simca}
\title{SIMCA one-class classification}
\usage{
simca(x, classname, ncomp = 15, center = T, scale = F, cv = NULL,
  exclcols = NULL, exclrows = NULL, x.test = NULL, c.test = NULL,
  alpha = 0.05, method = "svd", info = "")
}
\arguments{
\item{x}{a numerical matrix with data values.}

\item{classname}{short text (up to 20 symbols) with class name.}

\item{ncomp}{maximum number of components to calculate.}

\item{center}{logical, do mean centering of data or not.}

\item{scale}{logical, do sdandardization of data or not.}

\item{cv}{number of segments for random cross-validation (1 for full cross-validation).}

\item{exclcols}{columns to be excluded from calculations (numbers, names or vector with logical values)}

\item{exclrows}{rows to be excluded from calculations (numbers, names or vector with logical values)}

\item{x.test}{a numerical matrix with test data.}

\item{c.test}{a vector with classes of test data objects (can be text with names of classes or logical).}

\item{alpha}{significance level for calculating limit for T2 and Q residuals.}

\item{method}{method to compute principal components.}

\item{info}{text with information about the model.}
}
\value{
Returns an object of \code{simca} class with following fields:
\item{classname }{a short text with class name.} 
\item{modpower }{a matrix with modelling power of variables.} 
\item{calres }{an object of class \code{\link{simcares}} with classification results for a 
calibration data.} 
\item{testres }{an object of class \code{\link{simcares}} with classification results for a test 
data, if it was provided.} 
\item{cvres }{an object of class \code{\link{simcares}} with classification results for 
cross-validation, if this option was chosen.} 

Fields, inherited from \code{\link{pca}} class:
\item{ncomp }{number of components included to the model.} 
\item{ncomp.selected }{selected (optimal) number of components.} 
\item{loadings }{matrix with loading values (nvar x ncomp).} 
\item{eigenvals }{vector with eigenvalues for all existent components.} 
\item{expvar }{vector with explained variance for each component (in percent).} 
\item{cumexpvar }{vector with cumulative explained variance for each component (in percent).} 
\item{T2lim }{statistical limit for T2 distance.} 
\item{Qlim }{statistical limit for Q residuals.} 
\item{info }{information about the model, provided by user when build the model.}
}
\description{
\code{simca} is used to make SIMCA (Soft Independent Modelling of Class Analogies) model for 
one-class classification.
}
\details{
SIMCA is in fact PCA model with additional functionality, so \code{simca} class inherits most 
of the functionality of \code{\link{pca}} class.
}
\examples{
## make a SIMCA model for Iris setosa class with full cross-validation
library(mdatools)

data = iris[, 1:4]
class = iris[, 5]

# take first 20 objects of setosa as calibration set 
se = data[1:20, ]

# make SIMCA model and apply to test set
model = simca(se, 'setosa', cv = 1)
model = selectCompNum(model, 1)

# show infromation, summary and plot overview
print(model)
summary(model)
plot(model)

# show predictions 
par(mfrow = c(2, 1))
plotPredictions(model, show.labels = TRUE)
plotPredictions(model, res = 'calres', ncomp = 2, show.labels = TRUE)
par(mfrow = c(1, 1))

# show performance, modelling power and residuals for ncomp = 2
par(mfrow = c(2, 2))
plotSensitivity(model)
plotMisclassified(model)
plotModellingPower(model, ncomp = 2, show.labels = TRUE)
plotResiduals(model, ncomp = 2)
par(mfrow = c(1, 1))

}
\references{
S. Wold, M. Sjostrom. "SIMCA: A method for analyzing chemical data in terms of similarity and 
analogy" in B.R. Kowalski (ed.), Chemometrics Theory and Application, American Chemical Society 
Symposium Series 52, Wash., D.C., American Chemical Society, p. 243-282.
}
\seealso{
Methods for \code{simca} objects:
\tabular{ll}{
 \code{print.simca} \tab shows information about the object.\cr
 \code{summary.simca} \tab shows summary statistics for the model.\cr
 \code{plot.simca} \tab makes an overview of SIMCA model with four plots.\cr
 \code{\link{predict.simca}} \tab applies SIMCA model to a new data.\cr
 \code{\link{plotModellingPower.simca}} \tab shows plot with modelling power of variables.\cr
}

Methods, inherited from \code{classmodel} class:
\tabular{ll}{
 \code{\link{plotPredictions.classmodel}} \tab shows plot with predicted values.\cr
 \code{\link{plotSensitivity.classmodel}} \tab shows sensitivity plot.\cr
 \code{\link{plotSpecificity.classmodel}} \tab shows specificity plot.\cr
 \code{\link{plotMisclassified.classmodel}} \tab shows misclassified ratio plot.\cr
}

Methods, inherited from \code{\link{pca}} class:
\tabular{ll}{
 \code{\link{selectCompNum.pca}} \tab set number of optimal components in the model\cr
 \code{\link{plotScores.pca}} \tab shows scores plot.\cr
 \code{\link{plotLoadings.pca}} \tab shows loadings plot.\cr
 \code{\link{plotVariance.pca}} \tab shows explained variance plot.\cr
 \code{\link{plotCumVariance.pca}} \tab shows cumulative explained variance plot.\cr
 \code{\link{plotResiduals.pca}} \tab shows Q vs. T2 residuals plot.\cr
}
}

