% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CUSUMfixed.R
\name{CUSUMfixed}
\alias{CUSUMfixed}
\title{Self-normalized CUSUM tests for structural change under long memory.}
\usage{
CUSUMfixed(tseries, d, procedure, bandw, tau = 0.15)
}
\arguments{
\item{tseries}{the univariate numeric time series to be investigated.}

\item{d}{the long-memory parameter.}

\item{procedure}{specifies whether the CUSUM fixed-b or fixed-m type A or type B tests are used. It can be chosen between
\code{"CUSUMfixedb_typeA"}, \code{"CUSUMfixedb_typeB"}, \code{"CUSUMfixedm_typeA"}, and \code{"CUSUMfixedm_typeB"} (see Wenger, Leschinski (2019) for details).}

\item{bandw}{the bandwidth used for estimation of the long-run variance. For the fixed-b tests \code{b=[0.05,0.1,0.2,0.3,...,0.9,1]}, for the
fixed-m tests \code{m=[1,2,3,4,10,25,50,100,150,200]}. Recommended bandwidth by Wenger, Leschinski (2019) are \code{b=0.1} and \code{m=10}.}

\item{tau}{defines the search area, which is \code{[tau,1-tau]}. Default is \code{tau=0.15} as suggested by Andrews (1993).}
}
\value{
Returns the test statistic and the corresponding critical values of the test.
}
\description{
This function performs a family of CUSUM tests for a change-in-mean that are robust under long memory. They apply non-parametric kernel-based
fixed-b and fixed-m long-run variance estimators in the denominator of the test statistics.
The function returns the test statistic as well as critical values.
}
\details{
Note that the critical values are generated for \code{tau=0.15} using the Bartlett kernel for the fixed-b tests or averaging the first m periodogram
ordinates (which corresponds to the Daniell kernel) for the fixed-m tests.
}
\examples{
library(fracdiff)
library(longmemo)
library(LongMemoryTS)

n        <- 500
d        <- 0.2
tseries  <- fracdiff.sim(n=n,d=d)$series
d_est    <- local.W(tseries, m=floor(1+n^0.65))$d

changep  <- c(rep(0,n/2),rep(1,n/2))
tseries2 <- tseries+changep
d_est2   <- local.W(tseries2, m=floor(1+n^0.65))$d

CUSUMfixed(tseries,d=d_est,procedure="CUSUMfixedb_typeA",bandw=0.1)
CUSUMfixed(tseries,d=d_est,procedure="CUSUMfixedb_typeB",bandw=0.1)
CUSUMfixed(tseries,d=d_est,procedure="CUSUMfixedm_typeA",bandw=10)
CUSUMfixed(tseries,d=d_est,procedure="CUSUMfixedm_typeB",bandw=10)

CUSUMfixed(tseries2,d=d_est2,procedure="CUSUMfixedb_typeA",bandw=0.1)
CUSUMfixed(tseries2,d=d_est2,procedure="CUSUMfixedb_typeB",bandw=0.1)
CUSUMfixed(tseries2,d=d_est2,procedure="CUSUMfixedm_typeA",bandw=10)
CUSUMfixed(tseries2,d=d_est2,procedure="CUSUMfixedm_typeB",bandw=10)
}
\references{
Wenger, K. and Leschinski, C. (2019): Change-in-mean tests in long-memory time series: a review of recent developments. AStA Advances in Statistical Analysis, 103:2, pp. 237-256.

Hualde, J. and Iacone, F. (2017): Fixed bandwidth asymptotics for the studentized mean of fractionally integrated processes. Economics Letters, 150, pp. 39-43.

Andrews, D. W. K. (1993): Tests for Parameter Instability and Structural Change With Unknown Change Point. Econometrica, 61, pp. 821-856.
}
\author{
Kai Wenger
}
