% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pairwise.R
\name{pairwise}
\alias{pairwise}
\title{Transform meta-analysis data from two arm-based formats into
contrast-based format}
\usage{
pairwise(
  treat,
  event,
  n,
  mean,
  sd,
  TE,
  seTE,
  time,
  agent,
  dose,
  data = NULL,
  studlab,
  incr = gs("incr"),
  method = "Inverse",
  method.incr = gs("method.incr"),
  allstudies = gs("allstudies"),
  reference.group,
  keep.all.comparisons,
  sep.ag = "*",
  varnames = c("TE", "seTE"),
  append = !is.null(data),
  addincr = gs("addincr"),
  allincr = gs("allincr"),
  warn = FALSE,
  warn.deprecated = gs("warn.deprecated"),
  ...
)
}
\arguments{
\item{treat}{A list or vector with treatment information for
individual treatment arms (see Details).}

\item{event}{A list or vector with information on number of events
for individual treatment arms (see Details).}

\item{n}{A list or vector with information on number of
observations for individual treatment arms (see Details).}

\item{mean}{A list or vector with estimated means for individual
treatment arms (see Details).}

\item{sd}{A list or vector with information on the standard
deviation for individual treatment arms (see Details).}

\item{TE}{A list or vector with estimated treatment effects for
individual treatment arms (see Details).}

\item{seTE}{A list or vector with standard errors of estimated
treatment effect for individual treatment arms (see Details).}

\item{time}{A list or vector with information on person time at
risk for individual treatment arms (see Details).}

\item{agent}{A list or vector with agent information for
individual treatment arms (see Details).}

\item{dose}{A list or vector with dose information for
individual treatment arms (see Details).}

\item{data}{An optional data frame containing the study
information.}

\item{studlab}{A vector with study labels (optional).}

\item{incr}{A numerical value which is added to cell frequencies
for studies with a zero cell count, see Details.}

\item{method}{A character string indicating which method is to be
used to calculate treatment estimates. Either \code{"Inverse"}
or \code{"Peto"} (only for binary outcome), can be abbreviated.}

\item{method.incr}{A character string indicating which continuity
correction method should be used (\code{"only0"},
\code{"if0all"}, or \code{"all"}), see \code{\link{metabin}}.}

\item{allstudies}{A logical indicating if studies with zero or all
events in two treatment arms are to be included in the
meta-analysis (applies only if \code{sm} is equal to \code{"RR"}
or \code{"OR"}).}

\item{reference.group}{Reference treatment (first treatment is used
if argument is missing).}

\item{keep.all.comparisons}{A logical indicating whether all
pairwise comparisons or only comparisons with the study-specific
reference group should be kept ('basic parameters').}

\item{sep.ag}{A character used as separator
between agent in dose to create treatment labels.}

\item{varnames}{Character vector of length 2 with the variable names for the
treatment estimate and its standard error; by default, "TE" and "seTE".}

\item{append}{Either a logical indicating whether variables from the dataset
provided in argument \code{data} are appended to the dataset with
pairwise comparisons or a character vector with variable names to append to
the dataset.}

\item{addincr}{Deprecated argument (replaced by 'method.incr');
see \code{\link{metabin}}.}

\item{allincr}{Deprecated argument (replaced by 'method.incr');
see \code{\link{metabin}}.}

\item{warn}{A logical indicating whether warnings should be printed
(e.g., if studies are excluded due to only providing a single
treatment arm).}

\item{warn.deprecated}{A logical indicating whether warnings should
be printed if deprecated arguments are used.}

\item{\dots}{Additional arguments passed-through to the functions
to calculate effects.}
}
\value{
A data frame with the following columns:
\item{TE}{Treatment estimate comparing treatment 'treat1' and
  'treat2'.}
\item{seTE}{Standard error of treatment estimate.}
\item{studlab}{Study labels.}
\item{treat1}{First treatment in comparison.}
\item{treat2}{Second treatment in comparison.}
\item{event1}{Number of events for first treatment arm (for metabin
  and metainc).}
\item{event2}{Number of events for second treatment arm (for
  metabin and metainc).}
\item{n1}{Number of observations for first treatment arm (for
  metabin and metacont).}
\item{n2}{Number of observations for second treatment arm (for
  metabin and metacont).}
\item{mean1}{Estimated mean for first treatment arm (for
  metacont).}
\item{mean2}{Estimated mean for second treatment arm (for
  metacont).}
\item{sd1}{Standard deviation for first treatment arm (for
  metacont).}
\item{sd2}{Standard deviation for second treatment arm (for
  metacont).}
\item{TE1}{Estimated treatment effect for first treatment arm (for
  metagen).}
\item{TE2}{Estimated treatment effect for second treatment arm (for
  metagen).}
\item{seTE1}{Standard error of estimated treatment effect for first
  treatment arm (for metagen).}
\item{seTE2}{Standard error of estimated treatment effect for
  second treatment arm (for metagen).}
\item{time1}{Person time at risk for first treatment arm (for
  metainc).}
\item{time2}{Person time at risk for second treatment arm (for
  metainc).}
\item{agent1}{First agent in comparison.}
\item{agent2}{Second agent in comparison.}
\item{dose1}{Dose of first agent in comparison.}
\item{dose2}{Dose of second agent in comparison.}

All variables from the original dataset are also part of the output
dataset; see Details.
}
\description{
This function transforms data that are given in wide or long
arm-based format (e.g. input format for WinBUGS) to a
contrast-based format that is needed, for example, as input to R function
\code{\link[netmeta]{netmeta}} from R package \bold{netmeta}. The function
can transform data with binary, continuous, or generic outcomes as well as
incidence rates from arm-based to contrast-based format.
}
\details{
The pairwise function transforms data given in (wide or long)
arm-based format into the contrast-based format which consists of
\emph{pairwise} comparisons and is needed, for example, as input to the
\code{\link[netmeta]{netmeta}} function.

The pairwise function can transform data with binary outcomes
(using the \code{\link{metabin}} function from R package meta),
continuous outcomes (\code{\link{metacont}} function), incidence
rates (\code{\link{metainc}} function), and generic outcomes
(\code{\link{metagen}} function). Depending on the outcome, the
following arguments are mandatory:
\itemize{
\item treat, event, n (see \code{\link{metabin}});
\item treat, n, mean, sd (see \code{\link{metacont}});
\item treat, event, time (see \code{\link{metainc}});
\item treat, TE, seTE (see \code{\link{metagen}}).
}

Argument \code{treat} is mandatory to identify the individual
treatments. The other arguments contain outcome specific
data. These arguments must be either lists (wide arm-based format,
i.e., one row per study) or vectors (long arm-based format,
i.e., multiple rows per study) of the same length.

For the wide arm-based format, each list consists of as many
vectors of the same length as the multi-arm study with the largest
number of treatments. If a single multi-arm study has five arms,
five vectors have to be provided for each lists. Two-arm studies
have entries with \code{NA} for the third and subsequent
vectors. Each list entry is a vector with information for each
individual study; i.e., the length of this vector corresponds to
the total number of studies incorporated in the network
meta-analysis. Typically, list elements are part of a data frame
(argument \code{data}, optional); see Examples. An optional vector
with study labels can be provided which can be part of the data
frame.

In the long arm-based format, argument \code{studlab} is mandatory
to identify rows contributing to individual studies.

Additional arguments for meta-analysis functions can be provided
using argument '\dots'. The most important argument is \code{sm}
defining the summary measure. More information on this and other
arguments is given in the help pages of R functions
\code{\link{metabin}}, \code{\link{metacont}},
\code{\link{metainc}}, and \code{\link{metagen}}, respectively.

For standardised mean differences (argument \code{sm = "SMD"}),
equations (4) and (5) in Crippa & Orsini (2016) are used to
calculated SMDs and standard errors. These equations guarantee
consistent SMDs and standard errors for multi-arm studies. Note,
the summary measure is actually Cohen's d as Hedges' g is not
consistent in multi-arm studies.

For binary outcomes, 0.5 is added to all cell frequencies (odds
ratio) or only the number of events (risk ratio) for studies with a
zero cell count. For odds ratio and risk ratio, treatment estimates
and standard errors are only calculated for studies with zero or
all events in both groups if \code{allstudies} is \code{TRUE}. This
continuity correction is used both to calculate individual study
results with confidence limits and to conduct meta-analysis based
on the inverse variance method. For the risk difference, 0.5 is
only added to all cell frequencies to calculate the standard error.

For incidence rates, 0.5 is added to all cell frequencies for the
incidence rate ratio as summary measure. For the incidence risk
difference, 0.5 is only added to all cell frequencies to calculate
the standard error.

The value of pairwise is a data frame with as many rows as there
are pairwise comparisons. For each study with \emph{p} treatments,
\emph{p*(p-1) / 2} contrasts are generated. Each row contains the
treatment effect (\code{TE}), its standard error (\code{seTE}), the
treatments compared ((\code{treat1}), (\code{treat2})) and the
study label ((\code{studlab})). Further columns are added according
to type of data.

All variables from the original dataset are also part of the output
dataset if argument \code{append = TRUE}. If data are provided in the long
arm-based format, the value of a variable can differ between treatment arms;
for example, the mean age or percentage of women in the treatment arm. In
this situation, two variables instead of one variable will be included
in the output dataset. The values "1" and "2" are added to the
names for these variables, e.g. "mean.age1" and "mean.age2" for the
mean age.

In general, any variable names in the original dataset that are
identical to the main variable names (i.e., "TE", "seTE", ...) will
be renamed to variable names with ending ".orig".

A reduced dataset with basic comparisons (Rücker & Schwarzer,
2014) can be generated using argument \code{keep.all.comparisons =
FALSE}. Furthermore, the reference group for the basic comparisons
can be specified with argument \code{reference.group}.

\subsection{Use in network meta-analysis}{
R function \code{\link[netmeta]{netmeta}} expects data in a
\bold{contrast-based format}, where each row corresponds to a
comparison of two treatments and contains a measure of the
treatment effect comparing two treatments with standard error,
labels for the two treatments and an optional study label.  In
contrast-based format, a three-arm study contributes three rows
with treatment comparison and corresponding standard error for
pairwise comparison \emph{A} vs \emph{B}, \emph{A} vs \emph{C}, and
\emph{B} vs \emph{C} whereas a four-arm study contributes six rows
/ pairwise comparisons: \emph{A} vs \emph{B}, \emph{A} vs \emph{C},
\dots{}, \emph{C} vs \emph{D}.

Other programs for network meta-analysis in WinBUGS and Stata
require data in an \emph{arm-based} format, i.e. treatment estimate
for each treatment arm instead of a difference of two treatments. A
common \bold{(wide) arm-based format} consists of one data row per
study, containing treatment and other necessary information for all
study arms. For example, a four-arm study contributes one row with
four treatment estimates and corresponding standard errors for
treatments \emph{A}, \emph{B}, \emph{C}, and \emph{D}.  Another
possible arm-based format is a long format where each row
corresponds to a single study arm. Accordingly, in the \bold{long
arm-based format} a study contributes as many rows as treatments
considered in the study.
}
}
\note{
This function must not be confused with \code{\link[netmeta]{netpairwise}}
which can be used to conduct pairwise meta-analyses for all
comparisons with direct evidence in a network meta-analysis.
}
\examples{
p0 <- pairwise(studlab = study, treat = treatment,
  n = ni, mean = mi, sd = sdi, data = dat.senn2013,
  append = c("study", "comment"))
head(p0)
# Meta-analysis of studies comparing metformin to placebo
metagen(p0, subset = treat1 == "metformin" & treat2 == "placebo")

\dontrun{
# Use pairwise() to run network meta-analyses
# (R package 'netmeta' must be available)
if (requireNamespace("netmeta", quietly = TRUE)) {
 # Example using continuous outcomes (internal call of function
 # metacont)
 #
 Franchini2012 <- dat.franchini2012
 # Transform data from arm-based format to contrast-based format
 p1 <- pairwise(list(Treatment1, Treatment2, Treatment3),
   n = list(n1, n2, n3),
   mean = list(y1, y2, y3), sd = list(sd1, sd2, sd3),
   data = Franchini2012, studlab = Study)
 p1

 # Conduct network meta-analysis
 library("netmeta")
 #
 net1 <- netmeta(p1)
 net1

 # Draw network graphs
 #
 netgraph(net1, points = TRUE, cex.points = 3, cex = 1.5,
   thickness = "se.common")
 netgraph(net1, points = TRUE, cex.points = 3, cex = 1.5,
   plastic = TRUE, thickness = "se.common",
   iterate = TRUE)
 netgraph(net1, points = TRUE, cex.points = 3, cex = 1.5,
   plastic = TRUE, thickness = "se.common",
   iterate = TRUE, start = "eigen")

 # Example using generic outcomes (internal call of function
 # metagen)
 #
 # Calculate standard error for means y1, y2, y3
 Franchini2012$se1 <- with(Franchini2012, sqrt(sd1^2 / n1))
 Franchini2012$se2 <- with(Franchini2012, sqrt(sd2^2 / n2))
 Franchini2012$se3 <- with(Franchini2012, sqrt(sd3^2 / n3))
 # Transform data from arm-based format to contrast-based format
 # using means and standard errors (note, argument 'sm' has to be
 # used to specify that argument 'TE' is a mean difference)
 p2 <- pairwise(list(Treatment1, Treatment2, Treatment3),
   TE = list(y1, y2, y3), seTE = list(se1, se2, se3),
   n = list(n1, n2, n3),
   data = Franchini2012, studlab = Study,
   sm = "MD")
 p2

 # Compare pairwise objects p1 (based on continuous outcomes) and p2
 # (based on generic outcomes)
 #
 all.equal(
   p1[, c("TE", "seTE", "studlab", "treat1", "treat2")],
   p2[, c("TE", "seTE", "studlab", "treat1", "treat2")])

 # Same result as network meta-analysis based on continuous outcomes
 # (object net1)
 net2 <- netmeta(p2)
 net2

 # Example with binary data
 #
 data(smokingcessation)
 # Transform data from arm-based format to contrast-based format
 # (internal call of metabin function). Argument 'sm' has to be used
 # for odds ratio as risk ratio (sm = "RR") is default of metabin
 # function.
 #
 p3 <- pairwise(list(treat1, treat2, treat3),
   list(event1, event2, event3), list(n1, n2, n3),
   data = smokingcessation,
   sm = "OR")
 p3

 # Conduct network meta-analysis
 #
 net3 <- netmeta(p3)
 net3

 # Example with incidence rates
 #
 data(dietaryfat)

 # Transform data from arm-based format to contrast-based format
 #
 p4 <- pairwise(list(treat1, treat2, treat3),
   list(d1, d2, d3), time = list(years1, years2, years3),
   studlab = ID,
   data = dietaryfat)
 p4

 # Conduct network meta-analysis using incidence rate ratios (sm =
 # "IRR"). Note, the argument 'sm' is not necessary as this is the
 # default in R function metainc called internally.
 #
 net4 <- netmeta(p4, sm = "IRR")
 summary(net4)

 # Example with long data format
 #
 # Transform data from long arm-based format to contrast-based
 # format Argument 'sm' has to be used for odds ratio as summary
 # measure; by default the risk ratio is used in the metabin
 # function called internally.
 #
 p5 <- pairwise(treatment, event = r, n = N,
   studlab = author, data = dat.woods2010, sm = "OR")
 p5

 # Conduct network meta-analysis
 net5 <- netmeta(p5)
 net5
}
}

}
\references{
Crippa A, Orsini N (2016):
Dose-response meta-analysis of differences in means.
\emph{BMC Medical Research Methodology},
\bold{16}:91.
}
\seealso{
\code{\link{longarm}}, \code{\link{metabin}},
  \code{\link{metacont}}, \code{\link{metagen}},
  \code{\link{metainc}}, \code{\link[netmeta]{netmeta}},
  \code{\link[netmeta]{netgraph.netmeta}},
  \code{\link[metadat]{dat.senn2013}},
  \code{\link[metadat]{dat.franchini2012}},
  \code{\link[metadat]{dat.franchini2012}}
}
\author{
Gerta Rücker\email{gerta.ruecker@uniklinik-freiburg.de}, Guido
  Schwarzer \email{guido.schwarzer@uniklinik-freiburg.de}
}
\keyword{datagen}
