\name{Metacommunity}
\alias{Metacommunity}


\title{
Analysis of the "Elements of Metacommunity Structure"
}

\description{
\code{Metacommunity} is a wrapper for the analysis of the three elements of metacommunity structure (coherence, boundary clumping, & turnover) following the framework of Leibold & Mikkelson 2002. It is important to note here that the results of boundary clumping and turnover are only relevant if the matrix is significantly positively coherent (i.e. empirical matrix has fewer embedded absences than null matrices).

}

\usage{
Metacommunity(comm, scores=1, method = "r1", sims = 1000, 
  order=TRUE, allow.empty=FALSE, binary=TRUE, verbose=FALSE,  
  modularity=FALSE, c = sum(dim(comm)), nstarts= 10)

}



\arguments{
  \item{comm}{
community data in the form of a presence absence matrix
}
 
  \item{scores}{
Axis scores to ordinate matrix. 1: primary axis scores (default) 2: secondary axis scores. See Details.
}

 \item{method}{
null model randomization method used by \code{NullMaker}. See details below (and the help file of fucntion \code{NullMaker}) for more information. 
}
  \item{sims}{
number of simulated null matrices to use in analysis
}

		\item{order}{
logical argument indicating whether to ordinate the interaction matrix or not. See details.
}

		\item{allow.empty}{
logical argument indicating whether to allow null matrices to have empty rows or columns
}

 \item{binary}{
logical argument indicating whether to ordinate the community matrix 
based on abundance or binary (default) data.
}

 \item{verbose}{
logical. Prints a graphical progress bar that tracks the creation of null matrices. Useful for conservative null models on large and/or sparse data.
}

 \item{modularity}{
logical. Use Barber's modularity (`Modularity` function) as a measure of boundary clumping, or use Morisita's index (`BoundaryClump` function)?
}

	\item{c}{starting guess for the number of modules present. Defaults to the maximum number of modules possible.}
	
	\item{nstarts}{number of starts. Default is 100. More will both slow the function down, and increase the likelihood of converging on the true modularity value.}



}


\details{
\code{method} is an argument handed to functions in the \code{vegan} package. Leibold & Mikkelson advocated the use of equiprobable rows and columns (provided that rows and columns had at least one entry). This method is called \code{r00}. Methods maintaining row (site) frequencies include \code{r0},\code{r1}, and \code{r2}. The default method argument is \code{r1}, which maintains the species richness of a site (row totals) and fills species ranges (columns) based on their marginal probabilities. Arguably the most conservative null algorithm is the fixed row - fixed column total null, which can be attained using many of swap algorithms described in the vegan package (sequential methods like \code{tswap}, \code{swap}, and non-sequential \code{quasiswap} and \code{backtracking}). See the help file for \code{commsim} or Wright et al. 1998 for more information. 

If \code{order} is FALSE, the interaction matrix is not ordinated, allowing the user to order the matrix based on site characteristics or other biologically relevant characteristics.

}

\note{This function may take awhile to finish as a result of the creation of null matrices. If you are running multiple interaction matrices, the code can be parallelized using the \code{snow} package.}

\note{The progress bar displays for \code{NullMaker} function only. Large matrices for coherence calculations may cause a longer delay in between when the bar is finished and when the function is actually finished running.}


\value{
\item{Comm}{ordinated matrix used to calculate coherence, boundary clumping & turnover}

\item{Coherence}{output of the \code{Coherence} function, giving information on the number of embedded absences and the significance relative to simulated null matrices}

\item{Turnover}{output of the \code{Turnover} function, testing the number of species replacements relative to simulated null matrices}

\item{Boundary}{output of the \code{BoundaryClump} function, which calculates the Morisita's index, assessing significance using a chi-squared test}

}



\references{
Dallas,T. 2014. metacom: an R package for the analysis of metacommunity structure. Ecography. DOI:10.1111/j.1600-0587.2013.00695.x

Leibold, M.A. and G.M. Mikkelson. 2002. Coherence, species turnover, and boundary clumping: elements of meta-community structure. Oikos 97: 237 - 250.

Presley, S. J., C. L. Higgins, and M. R. Willig. 2010. A comprehensive framework for the evaluation of metacommunity structure. Oikos 119:908-917

Wright, D.H., Patterson, B.D., Mikkelson, G.M., Cutler, A. & Atmar, W. (1998). A comparative analysis of nested subset patterns of species composition. Oecologia 113, 1-20.
}


\author{
Tad Dallas
}


\examples{
#define an interaction matrix
data(TestMatrices)
intmat=TestMatrices[[5]]

#determines the elements of metacommunity structure 
ems.intmat=Metacommunity(intmat, method='r1', sims=100, scores=1)

#return results in list form
ems.intmat
}


