% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measure_diffusion.R
\name{node_diffusion}
\alias{node_diffusion}
\alias{node_adoption_time}
\alias{node_adopter}
\alias{node_thresholds}
\alias{node_infection_length}
\alias{node_exposure}
\title{Measures of nodes in a diffusion}
\usage{
node_adoption_time(diff_model)

node_adopter(diff_model)

node_thresholds(diff_model)

node_infection_length(diff_model)

node_exposure(.data, mark, time = 0)
}
\arguments{
\item{diff_model}{A valid network diffusion model,
as created by \code{as_diffusion()} or \code{play_diffusion()}.}

\item{.data}{An object of a \code{{manynet}}-consistent class:
\itemize{
\item matrix (adjacency or incidence) from \code{{base}} R
\item edgelist, a data frame from \code{{base}} R or tibble from \code{{tibble}}
\item igraph, from the \code{{igraph}} package
\item network, from the \code{{network}} package
\item tbl_graph, from the \code{{tidygraph}} package
}}

\item{mark}{A valid 'node_mark' object or
logical vector (TRUE/FALSE) of length equal to
the number of nodes in the network.}

\item{time}{A time point until which infections/adoptions should be
identified. By default \code{time = 0}.}
}
\description{
These functions allow measurement of various features of
a diffusion process:
\itemize{
\item \code{node_adoption_time()}: Measures the number of time steps until
nodes adopt/become infected
\item \code{node_adopter()}: Classifies membership of nodes into diffusion categories
\item \code{node_thresholds()}: Measures nodes' thresholds from the amount
of exposure they had when they became infected
\item \code{node_infection_length()}: Measures the average length nodes that become
infected remain infected in a compartmental model with recovery
\item \code{node_exposure()}: Measures how many exposures nodes have to
a given mark
\item \code{node_is_exposed()}: Marks the nodes that are susceptible,
i.e. are in the immediate neighbourhood of given mark vector
}
}
\section{Adoption time}{

\code{node_adoption_time()} measures the time units it took
until each node became infected.
Note that an adoption time of 0 indicates that this was a seed node.
}

\section{Adopter class}{

\code{node_adopter()} classifies the nodes involved in a diffusion
by where on the distribution of adopters they fell.
Valente (1995) defines five memberships:
\itemize{
\item \emph{Early adopter}: those with an adoption time less than
the average adoption time minus one standard deviation of adoptions times
\item \emph{Early majority}: those with an adoption time between
the average adoption time and
the average adoption time minus one standard deviation of adoptions times
\item \emph{Late majority}: those with an adoption time between
the average adoption time and
the average adoption time plus one standard deviation of adoptions times
\item \emph{Laggard}: those with an adoption time greater than
the average adoption time plus one standard deviation of adoptions times
\item \emph{Non-adopter}: those without an adoption time,
i.e. never adopted
}
}

\section{Thresholds}{

\code{node_thresholds()} infers nodes' thresholds based on how much
exposure they had when they were infected.
This inference is of course imperfect,
especially where there is a sudden increase in exposure,
but it can be used heuristically.
}

\section{Infection length}{

\code{node_infection_length()} measures the average length of time that nodes
that become infected remain infected in a compartmental model with recovery.
Infections that are not concluded by the end of the study period are
calculated as infinite.
}

\section{Exposure}{

\code{node_exposure()} calculates the number of infected/adopting nodes
to which each susceptible node is exposed.
It usually expects network data and
an index or mark (TRUE/FALSE) vector of those nodes which are currently infected,
but if a diff_model is supplied instead it will return
nodes exposure at \eqn{t = 0}.
}

\examples{
  smeg <- manynet::generate_smallworld(15, 0.025)
  smeg_diff <- play_diffusion(smeg, recovery = 0.2)
  plot(smeg_diff)
  # To measure when nodes adopted a diffusion/were infected
  (times <- node_adoption_time(smeg_diff))
  # To classify nodes by their position in the adoption curve
  (adopts <- node_adopter(smeg_diff))
  summary(adopts)
  summary(times, membership = adopts)
  # To infer nodes' thresholds
  node_thresholds(smeg_diff)
  # To measure how long each node remains infected for
  node_infection_length(smeg_diff)
  # To measure how much exposure nodes have to a given mark
  node_exposure(smeg, mark = c(1,3))
  node_exposure(smeg_diff)
}
\references{
Valente, Tom W. 1995. \emph{Network models of the diffusion of innovations}
(2nd ed.). Cresskill N.J.: Hampton Press.
}
\seealso{
Other measures: 
\code{\link{between_centrality}},
\code{\link{close_centrality}},
\code{\link{closure}},
\code{\link{cohesion}()},
\code{\link{degree_centrality}},
\code{\link{eigenv_centrality}},
\code{\link{features}},
\code{\link{heterogeneity}},
\code{\link{hierarchy}},
\code{\link{holes}},
\code{\link{net_diffusion}},
\code{\link{periods}}

Other diffusion: 
\code{\link{net_diffusion}}
}
\concept{diffusion}
\concept{measures}
