#' Displays item surface and information plots
#' 
#' \code{itemplot} displays various item based IRT plots.
#' 
#'
#' @usage 
#' itemplot(object, ...)
#' 
#' \S4method{itemplot}{mirtClass}(object, item, type = 'trace', ...)
#'
#' \S4method{itemplot}{bfactorClass}(object, item, type = 'trace' , ...) 
#'
#' @aliases itemplot-method itemplot,mirtClass-method 
#' itemplot,bfactorClass-method
#' @param object a computed model of class \code{bfactorClass} or
#' \code{mirtClass}
#' @param item a single numeric value indicating which item to plot
#' @param type plot type to use, information (\code{'info'}), information contours \code{('infocontour')},
#'  or item trace lines (\code{'trace'})
#' @param ... additional arguments to be passed to lattice 
#' @author Phil Chalmers \email{rphilip.chalmers@@gmail.com}
#' @seealso \code{\link{plot}}, \code{\link{mirt}}, \code{\link{bfactor}},
#' \code{\link{polymirt}}
#' @rdname itemplot-methods  
#' @keywords plot
#' @docType methods
#' @exportMethod itemplot
#' @examples
#' 
#' \dontrun{
#' 
#' data(LSAT7)
#' fulldata <- expand.table(LSAT7)
#' mod1 <- mirt(fulldata,1)
#' 
#' itemplot(mod1, 2)
#'     }
#' 
setGeneric("itemplot", 
           def = function(object, ...) standardGeneric("itemplot")
)

#------------------------------------------------------------------------------
# Methods for Function itemplot
# 
# Plot individual items for fitted \code{mirt}, \code{bfactor}, or
# \code{polymirt} models.
# 
# 
# @name itemplot
# @docType methods
# @rdname itemplot-methods  
#' @export itemplot
# @keywords methods
setMethod(
	f = "itemplot",
	signature = signature(object = 'mirtClass'),
	definition = function(object, item, type = 'trace', ...)
	{  			
		x <- itemplot.main(object, item, type, ...)		        
		return(x)
	}
)

#------------------------------------------------------------------------------
# @rdname itemplot-methods  
setMethod(
	f = "itemplot",
	signature = signature(object = 'bfactorClass'),
	definition = function(object, item, type = 'trace', ...)
	{
	    x <- itemplot.main(object, item, type, ...)    	
	    return(x)
	}
)

itemplot.main <- function(x, item, type, ...){    
    nfact <- ncol(x@F)
    if(nfact > 2 && !x[[1]]@bfactor) stop('Can not plot high dimensional models')
    Theta <- x@Theta    
    P <- ProbTrace(x=x@pars[[item]], Theta=Theta)     
    a <- ExtractLambdas(x@pars[[item]])
    if(x@pars[[item]]@bfactor) a <- a[x@pars[[item]]@est[1:nfact]]
    A <- sqrt(sum(a^2))
    info <- ItemInfo(x=x@pars[[item]], A, Theta=Theta)
    if(nfact == 1){
        if(type == 'trace'){
            P <- ProbTrace(x=x@pars[[item]], Theta=Theta)
            plot(Theta, P[,1], col = 1, type='l', main = paste('Item', item), 
                 ylab = expression(P(theta)), xlab = expression(theta), ylim = c(0,1), ...)
            for(i in 2:ncol(P))
                lines(Theta, P[,i], col = i)                 
        }
        if(type == 'info'){            
            plot(Theta, info, col = 1, type='l', main = paste('Information for item', item), 
                 ylab = expression(I(theta)), xlab = expression(theta))
        }
        if(type == 'infocontour') stop('Cannot draw contours for 1 factor models')
    } else {
        plt <- data.frame(info = info, Theta1 = Theta[,1], Theta2 = Theta[,2])
        plt2 <- data.frame(P = P, Theta1 = Theta[,1], Theta2 = Theta[,2])
        colnames(plt2) <- c(paste("P", 1:ncol(P), sep=''), "Theta1", "Theta2")
        plt2 <- reshape(plt2, direction='long', varying = paste("P", 1:ncol(P), sep=''), v.names = 'P', 
                times = paste("P", 1:ncol(P), sep=''))
        colnames(plt) <- c("info", "Theta1", "Theta2")            
        if(type == 'infocontour')												
            return(contourplot(info ~ Theta1 * Theta2, data = plt, 
                               main = paste("Item", item, "Information Contour"), xlab = expression(theta[1]), 
                               ylab = expression(theta[2])), ...)
        if(type == 'info')
            return(lattice::wireframe(info ~ Theta1 + Theta2, data = plt, main = paste("Item ", item, "Information"), 
                             zlab=expression(I(theta)), xlab=expression(theta[1]), ylab=expression(theta[2]), 
                             scales = list(arrows = FALSE), colorkey = TRUE, drape = TRUE, ...))
        if(type == 'trace'){
            return(lattice::wireframe(P ~ Theta1 + Theta2|time, data = plt2, main = paste("Item ", item, "Trace"), 
                             zlab=expression(I(theta)), xlab=expression(theta[1]), ylab=expression(theta[2]), 
                             scales = list(arrows = FALSE), colorkey = TRUE, drape = TRUE, ...))            
        }        
    }    
}
