% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/test.welch.R
\name{test.welch}
\alias{test.welch}
\title{Welch's Test}
\usage{
test.welch(formula, data, alternative = c("two.sided", "less", "greater"),
           posthoc = TRUE, conf.level = 0.95, hypo = TRUE, descript = TRUE,
           effsize = FALSE, weighted = FALSE, ref = NULL, correct = FALSE,
           plot = FALSE, point.size = 4, adjust = TRUE, error.width = 0.1,
           xlab = NULL, ylab = NULL, ylim = NULL, breaks = ggplot2::waiver(),
           jitter = TRUE, jitter.size = 1.25, jitter.width = 0.05,
           jitter.height = 0, jitter.alpha = 0.1, title = "",
           subtitle = "Confidence Interval", digits = 2, p.digits = 4,
           as.na = NULL, write = NULL, append = TRUE, check = TRUE,
           output = TRUE, ...)
}
\arguments{
\item{formula}{a formula of the form \code{y ~ group} where \code{y} is
a numeric variable giving the data values and \code{group}
a numeric variable, character variable or factor with two
or more than two values or factor levels giving the
corresponding groups.}

\item{data}{a matrix or data frame containing the variables in the
formula \code{formula}.}

\item{alternative}{a character string specifying the alternative hypothesis,
must be one of \code{"two.sided"} (default), \code{"greater"}
or \code{"less"}. Note that this argument is only used when
conducting Welch's two-sample t-test.}

\item{posthoc}{logical: if \code{TRUE}, Games-Howell post hoc test for
multiple comparison is conducted when performing Welch's
ANOVA.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence
level of the interval.}

\item{hypo}{logical: if \code{TRUE} (default), null and alternative hypothesis
are shown on the console.}

\item{descript}{logical: if \code{TRUE} (default), descriptive statistics are shown
on the console.}

\item{effsize}{logical: if \code{TRUE}, effect size measure Cohen's d for
Welch's two-sample t-test (see \code{\link{cohens.d}}),
\eqn{\eta^2} and \eqn{\omega^2} for Welch's ANOVA and
Cohen's d for the post hoc tests are shown on the console.}

\item{weighted}{logical: if \code{TRUE}, the weighted pooled standard
deviation is used to compute Cohen's d.}

\item{ref}{a numeric value or character string indicating the reference
group. The standard deviation of the reference group is used
to standardized the mean difference to compute Cohen's d.}

\item{correct}{logical: if \code{TRUE}, correction factor to remove positive
bias in small samples is used.}

\item{plot}{logical: if \code{TRUE}, a plot showing error bars for
confidence intervals is drawn.}

\item{point.size}{a numeric value indicating the \code{size} aesthetic for
the point representing the mean value.}

\item{adjust}{logical: if \code{TRUE} (default), difference-adjustment for the
confidence intervals is applied.}

\item{error.width}{a numeric value indicating the horizontal bar width of
the error bar.}

\item{xlab}{a character string specifying the labels for the x-axis.}

\item{ylab}{a character string specifying the labels for the y-axis.}

\item{ylim}{a numeric vector of length two specifying limits of the
limits of the y-axis.}

\item{breaks}{a numeric vector specifying the points at which tick-marks
are drawn at the y-axis.}

\item{jitter}{logical: if \code{TRUE} (default), jittered data points
are drawn.}

\item{jitter.size}{a numeric value indicating the \code{size} aesthetic
for the jittered data points.}

\item{jitter.width}{a numeric value indicating the amount of horizontal jitter.}

\item{jitter.height}{a numeric value indicating the amount of vertical jitter.}

\item{jitter.alpha}{a numeric value indicating the opacity of the jittered
data points.}

\item{title}{a character string specifying the text for the title for
the plot.}

\item{digits}{an integer value indicating the number of decimal places
to be used for displaying descriptive statistics and
confidence interval.}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying the \emph{p}-value.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting
the analysis.}

\item{write}{a character string naming a text file with file extension
\code{".txt"} (e.g., \code{"Output.txt"}) for writing the
output into a text file.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown on the console.}

\item{...}{further arguments to be passed to or from methods.}

\item{subtitle}{a character string specifying the text for the subtitle for
the plot.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
  \item{\code{call}}{function call}
  \item{\code{type}}{type of analysis}
  \item{\code{sample}}{type of sample, i.e., two- or multiple sample}
  \item{\code{formula}}{formula of the current analysis}
  \item{\code{data}}{data frame specified in \code{data}}
  \item{\code{plot}}{ggplot2 object for plotting the results}
  \item{\code{args}}{specification of function arguments}
  \item{\code{result}}{result table}
}
\description{
This function performs Welch's two-sample t-test and Welch's ANOVA including
Games-Howell post hoc test for multiple comparison and provides descriptive
statistics, effect size measures, and a plot showing error bars for
difference-adjusted confidence intervals with jittered data points.
}
\details{
\describe{
\item{\strong{Effect Size Measure}}{By default, Cohen's d based on the non-weighted
standard deviation (i.e., \code{weighted = FALSE}) which does not assume homogeneity
of variance is computed (see Delacre et al., 2021) when requesting an effect size
measure (i.e., \code{effsize = TRUE}). Cohen's d based on the pooled standard
deviation assuming equality of variances between groups can be requested by
specifying \code{weighted = TRUE}.}
}
}
\examples{
dat1 <- data.frame(group1 = c(1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2),
                   group2 = c(1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3),
                   y = c(3, 1, 4, 2, 5, 3, 2, 3, 6, 6, 3, NA))

#-------------------------------------------------------------------------------
# Two-Sample Design

# Example 1a: Two-sided two-sample Welch-test
test.welch(y ~ group1, data = dat1)

# Example 1b: One-sided two-sample Welch-test
test.welch(y ~ group1, data = dat1, alternative = "greater")

# Example 1c: Two-sided two-sample Welch-test
# print Cohen's d with weighted pooled SD
test.welch(y ~ group1, data = dat1, effsize = TRUE)

# Example 1d: Two-sided two-sample Welch-test
# print Cohen's d with unweighted pooled SD
test.welch(y ~ group1, data = dat1, effsize = TRUE, weighted = FALSE)

# Example 1e: Two-sided two-sample Welch-test
# print Cohen's d with weighted pooled SD and
# small sample correction factor
test.welch(y ~ group1, data = dat1, effsize = TRUE, correct = TRUE)

# Example 1f: Two-sided two-sample Welch-test
# print Cohen's d with SD of the reference group 1
test.welch(y ~ group1, data = dat1, effsize = TRUE,
           ref = 1)

# Example 1g: Two-sided two-sample Welch-test
# print Cohen's d with weighted pooled SD and
# small sample correction factor
test.welch(y ~ group1, data = dat1, effsize = TRUE,
           correct = TRUE)

# Example 1h: Two-sided two-sample Welch-test
# do not print hypotheses and descriptive statistics,
test.welch(y ~ group1, data = dat1, descript = FALSE, hypo = FALSE)

# Example 1i: Two-sided two-sample Welch-test
# print descriptive statistics with 3 digits and p-value with 5 digits
test.welch(y ~ group1, data = dat1, digits = 3, p.digits = 5)

\dontrun{
# Example 1j: Two-sided two-sample Welch-test
# plot results
test.welch(y ~ group1, data = dat1, plot = TRUE)

# Load ggplot2 package
library(ggplot2)

# Save plot, ggsave() from the ggplot2 package
ggsave("Two-sample_Welch-test.png", dpi = 600, width = 4, height = 6)

# Example 1k: Two-sided two-sample Welch-test
# extract plot
p <- test.welch(y ~ group1, data = dat1, output = FALSE)$plot
p

# Extract data
plotdat <- test.welch(y ~ group1, data = dat1, output = FALSE)$data

# Draw plot in line with the default setting of test.welch()
ggplot(plotdat, aes(factor(group), y)) +
  geom_point(stat = "summary", fun = "mean", size = 4) +
  stat_summary(fun.data = "mean_cl_normal", geom = "errorbar", width = 0.20) +
  scale_x_discrete(name = NULL) +
  labs(subtitle = "Two-Sided 95% Confidence Interval") +
  theme_bw() + theme(plot.subtitle = element_text(hjust = 0.5))
}
#-------------------------------------------------------------------------------
# Multiple-Sample Design

# Example 2a: Welch's ANOVA
test.welch(y ~ group2, data = dat1)

# Example 2b: Welch's ANOVA
# print eta-squared and omega-squared
test.welch(y ~ group2, data = dat1, effsize = TRUE)

# Example 2c: Welch's ANOVA
# do not print hypotheses and descriptive statistics,
test.welch(y ~ group2, data = dat1, descript = FALSE, hypo = FALSE)

\dontrun{
# Example 2d: Welch's ANOVA
# plot results
test.welch(y ~ group2, data = dat1, plot = TRUE)

# Load ggplot2 package
library(ggplot2)

# Save plot, ggsave() from the ggplot2 package
ggsave("Multiple-sample_Welch-test.png", dpi = 600, width = 4.5, height = 6)

# Example 2e: Welch's ANOVA
# extract plot
p <- test.welch(y ~ group2, data = dat1, output = FALSE)$plot
p

# Extract data
plotdat <- test.welch(y ~ group2, data = dat1, output = FALSE)$data

# Draw plot in line with the default setting of test.welch()
ggplot(plotdat, aes(group, y)) +
  geom_point(stat = "summary", fun = "mean", size = 4) +
  stat_summary(fun.data = "mean_cl_normal", geom = "errorbar", width = 0.20) +
  scale_x_discrete(name = NULL) +
  labs(subtitle = "Two-Sided 95% Confidence Interval") +
  theme_bw() + theme(plot.subtitle = element_text(hjust = 0.5))
}
}
\references{
Rasch, D., Kubinger, K. D., & Yanagida, T. (2011). \emph{Statistics in psychology
- Using R and SPSS}. John Wiley & Sons.

Delacre, M., Lakens, D., Ley, C., Liu, L., & Leys, C. (2021). Why Hedges' g*s
based on the non-pooled standard deviation should be reported with Welch's t-test.
https://doi.org/10.31234/osf.io/tu6mp
}
\seealso{
\code{\link{test.t}}, \code{\link{test.z}}, \code{\link{test.levene}},
\code{\link{aov.b}}, \code{\link{cohens.d}}, \code{\link{ci.mean.diff}},
\code{\link{ci.mean}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
