% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/4_distances_cont.R
\name{hellinger.cont}
\alias{hellinger.cont}
\alias{hellinger.boot.cont}
\title{Estimation of a Continuous Mixture Complexity Based on Hellinger Distance}
\usage{
hellinger.cont(obj, bandwidth, j.max = 10, threshold = "SBC", sample.n = 5000,
               sample.plot = FALSE, control = c(trace = 0))

hellinger.boot.cont(obj, bandwidth, j.max = 10, B = 100, ql = 0.025,
                    qu = 0.975, sample.n = 3000, sample.plot = FALSE,
                    control = c(trace = 0), ...)
}
\arguments{
\item{obj}{object of class \code{\link{datMix}}.}

\item{bandwidth}{numeric, indicating the bandwidth to be used. Can also be set to "adaptive" if the adaptive kernel density estimator as defined by Cutler & Cordero-Brana (1996, page 1720, Equation 2) should be employed.}

\item{j.max}{integer stating the maximal number of components to be considered.}

\item{threshold}{function or character string in \code{c("AIC", "SBC")} specifying which threshold should be used to compare two mixture estimates of complexities \eqn{j} and \eqn{j+1}. If the difference in minimized squared distances is smaller than the relevant threshold, \eqn{j} will be returned as complexity estimate.}

\item{sample.n}{integer, specifying the sample size to be used for approximation of the objective function (see details).}

\item{sample.plot}{logical, indicating whether the histogram of the sample drawn to approximate the objective function should be plotted.}

\item{control}{control list of optimization parameters, see \code{\link[Rsolnp]{solnp}}.}

\item{B}{integer, specifying the number of bootstrap replicates.}

\item{ql}{numeric between \eqn{0} and \eqn{1}, specifying the lower quantile to which the observed difference in minimized squared distances will be compared.}

\item{qu}{numeric between \eqn{0} and \eqn{1}, specifying the upper quantile to which the observed difference in minimized squared distances will be compared.}

\item{\dots}{further arguments passed to the \code{\link[boot]{boot}} function.}
}
\value{
Object of class \code{paramEst} with the following attributes:
     \item{dat}{data based on which the complexity is estimated.}
     \item{dist}{character string stating the (abbreviated) name of the component distribution, such that the function \code{ddist} evaluates its density/ mass function and \code{rdist} generates random variates.}
     \item{ndistparams}{integer specifying the number of parameters identifying the component distribution, i.e. if \eqn{\theta} is in \eqn{R^d} then \code{ndistparams}\eqn{ = d}.}
     \item{formals.dist}{string vector specifying the names of the formal arguments identifying the distribution \code{dist} and used in \code{ddist} and \code{rdist}, e.g. for a gaussian mixture (\code{dist = norm}) amounts to 
     \code{mean} and \code{sd}, as these are the formal arguments used by \code{dnorm} and \code{rnorm}.}
     \item{discrete}{logical indicating whether the underlying mixture distribution is discrete. Will always be \code{FALSE} in this case.}
     \item{mle.fct}{attribute \code{MLE.function} of \code{obj}.}
     \item{pars}{say the complexity estimate is equal to some \eqn{j}. Then \code{pars} is a numeric vector of size \eqn{(d+1)*j-1} specifying the component weight and parameter estimates, given as 
\deqn{(w_1, ... w_{j-1}, \theta 1_1, ... \theta 1_j, \theta 2_1, ... \theta d_j).}}
     \item{values}{numeric vector of function values gone through during optimization at iteration \eqn{j}, the last entry being the value at the optimum.}
     \item{convergence}{integer, indicating whether the solver has converged (0) or not (1 or 2) at iteration \eqn{j}.}
}
\description{
Estimation of a continuous mixture complexity as well as its component weights and parameters by minimizing the squared Hellinger distance to a kernel density estimate.
}
\details{
Define the \eqn{complexity} of a finite continuous mixture \eqn{F} as the smallest integer \eqn{p}, such that its probability density function (pdf) \eqn{f} can be written as
\deqn{f(x) = w_1*g(x;\theta_1) + \dots + w_p*g(x;\theta_p).}
Further, let \eqn{g, f} be two probability density functions. The squared Hellinger distance between \eqn{g} and \eqn{f} is given by
\deqn{H^2(g,f) = \int (\sqrt{g(x)}-\sqrt{f(x)})^2 = 2 - 2\int\sqrt{f(x)}\sqrt{g(x)},}
where \eqn{\sqrt{g(x)}}, respectively \eqn{\sqrt{f(x)}} denotes the square root of the probability density functions at point x.
To estimate \eqn{p}, \code{hellinger.cont} iteratively increases the assumed complexity \eqn{j} and finds the "best" estimate for both, the pdf of a mixture with \eqn{j} and \eqn{j+1} components, ideally by calculating the parameters that minimize the sum of squared Hellinger distances to a kernel density estimate evaluated at each point.  Since the computational burden of optimizing over an integral to find the "best" component weights and parameters is immense, the algorithm approximates the objective function by sampling \code{sample.n} observations \eqn{Y_i} from the kernel density estimate and using
\deqn{2 - 2\sum \sqrt{f(Y_i)} / \sqrt{g(Y_i)},}
instead, with \eqn{f} being the mixture density and \eqn{g} being the kernel density estimate. Once the "best" parameters have been obtained, the difference in squared distances is compared to a predefined \code{threshold}. If this difference is smaller than the threshold, the algorithm terminates and the true complexity is estimated as \eqn{j}, otherwise \eqn{j} is increased by 1 and the procedure is started over. The predefined thresholds are the \code{"AIC"} given by 
\deqn{(d+1)/n}
and the \code{"SBC"} given by 
\deqn{(d+1)log(n)/(2n),} 
\eqn{n} being the sample size and \eqn{d} the number of component parameters, i.e. \eqn{\theta} is in \eqn{R^d}. Note that, if a customized function is to be used, it may only take the arguments \code{j} and \code{n}, so if the user wants to include the number of component parameters \eqn{d}, it has to be entered explicitly.
\code{hellinger.boot.cont} works similarly to \code{hellinger.cont} with the exception that the difference in squared distances is not compared to a predefined threshold but a value generated by a bootstrap procedure. At every iteration (\eqn{j}), the function sequentially tests \eqn{p = j} versus \eqn{p = j+1} for \eqn{j = 1,2, \dots}, using a parametric bootstrap to generate \code{B} samples of size \eqn{n} from a \eqn{j}-component mixture given the previously calculated "best" parameter values. For each of the bootstrap samples, again the "best" estimates corresponding to pdfs with \eqn{j} and \eqn{j+1} components are calculated, as well as their difference in approximated squared Hellinger distances from the kernel density estimate. The null hypothesis \eqn{H_0: p = j} is rejected and \eqn{j} increased by 1 if the original difference in squared distances lies outside of the interval \eqn{[ql, qu]}, specified by the \code{ql} and \code{qu} empirical quantiles of the bootstrapped differences. Otherwise, \eqn{j} is returned as the complexity estimate.
To calculate the minimum of the Hellinger distance (and the corresponding parameter values), the solver \code{\link[Rsolnp]{solnp}} is used. The initial values supplied to the solver are calculated as follows: the data is clustered into \eqn{j} groups by the function \code{\link[cluster]{clara}} and the data corresponding to each group is given to \code{MLE.function} (if supplied to the \code{\link{datMix}} object \code{obj}, otherwise numerical optimization is used here as well). The size of the groups is taken as initial component weights and the MLE's are taken as initial component parameter estimates.
}
\examples{

### generating 'Mix' object
normLocMix <- Mix("norm", discrete = FALSE, w = c(0.3, 0.4, 0.3), mean = c(10, 13, 17),
                  sd = c(1, 1, 1))

### generating 'rMix' from 'Mix' object (with 1000 observations)
set.seed(1)
normLocRMix <- rMix(10000, normLocMix)

### generating 'datMix' from 'R' object

## generate list of parameter bounds

norm.bound.list <- list("mean" = c(-Inf, Inf), "sd" = c(0, Inf))

## generate MLE functions

# for "mean"
MLE.norm.mean <- function(dat) mean(dat)
# for "sd" (the sd function uses (n-1) as denominator)
MLE.norm.sd <- function(dat){
sqrt((length(dat) - 1) / length(dat)) * sd(dat)
}
# combining the functions to a list
MLE.norm.list <- list("MLE.norm.mean" = MLE.norm.mean,
                      "MLE.norm.sd" = MLE.norm.sd)

## generating 'datMix' object
normLoc.dM <- RtoDat(normLocRMix, theta.bound.list = norm.bound.list,
                     MLE.function = MLE.norm.list)


### complexity and parameter estimation
\dontrun{
set.seed(0)
res <- hellinger.cont(normLoc.dM, bandwidth = 0.5, sample.n = 5000)
plot(res)
}

}
\references{
Details can be found in 
\enumerate{
\item M.-J. Woo and T. Sriram, "Robust Estimation of Mixture Complexity", Journal of the American Statistical Association, Vol. 101, No. 476, 1475-1486, Dec. 2006.
\item A. Cutler, O.I. Cordero-Brana, "Minimum Hellinger Distance Estimation for Finite Mixture Models." Journal of the American Statistical Association, Vol. 91, No. 436, 1716-1723,  Dec. 1996.}
}
\seealso{
\code{\link{hellinger.disc}} for the same estimation method for discrete mixtures, \code{\link[Rsolnp]{solnp}} for the solver, \code{\link{datMix}} for the creation of the \code{datMix} object.
}
\keyword{cluster}
