% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mlb_tier_maker.R
\name{mlb_player_tiers}
\alias{mlb_player_tiers}
\title{Create MLB Player Tiers}
\usage{
mlb_player_tiers(
  data,
  title = "MLB Player Tiers",
  subtitle = "Created with the #mlbplotR Tiermaker",
  caption = NULL,
  tier_desc = c(`1` = "MVP Candidates", `2` = "Very Good", `3` = "Medium", `4` = "Bad",
    `5` = "Negative WAR", `6` = "", `7` = ""),
  presort = FALSE,
  alpha = 1,
  width = 0.1,
  no_line_below_tier = NULL,
  devel = FALSE,
  background_color = "#1e1e1e",
  line_color = "#e0e0e0",
  title_color = "white",
  subtitle_color = "#8e8e93",
  caption_color = subtitle_color,
  tier_label_color = title_color,
  na_headshot_to_logo = TRUE
)
}
\arguments{
\item{data}{A data frame that has to include the variables \code{tier_no} (the
number of the tier starting from the top tier no. 1) and \code{player_id} (the
player's MLBAM/Savant ID). If data includes the variable \code{tier_rank},
these ranks will be used within each tier. Otherwise, if \code{presort = FALSE},
the function will assume that data is already sorted and if \code{presort = TRUE},
teams will be sorted alphabetically within tiers.}

\item{title}{The title of the plot. If \code{NULL}, it will be omitted.}

\item{subtitle}{The subtitle of the plot. If \code{NULL}, it will be omitted.}

\item{caption}{The caption of the plot. If \code{NULL}, it will be omitted.}

\item{tier_desc}{A named vector consisting of the tier descriptions. The names
must equal the tier numbers from \code{tier_no}}

\item{presort}{If \code{FALSE} (the default) the function assumes that the teams
are already sorted within the tiers. Will otherwise sort alphabetically.}

\item{alpha}{The alpha channel of the logos, i.e. transparency level, as a
numerical value between 0 and 1. Defaults to 1}

\item{width}{The desired width of the logo in \code{npc} (Normalised Parent Coordinates). A typical size is 0.1.}

\item{no_line_below_tier}{Vector of tier numbers. The function won't draw tier
separation lines below these tiers. This is intended to be used for tiers
that shall be combined (see examples).}

\item{devel}{Determines if headshots shall be rendered. If \code{FALSE} (the default),
headshots will be rendered on each run. If \code{TRUE} the player ids will be
plotted instead of the logos. This is much faster and helps with the plot
development.}

\item{background_color}{Background color for the plot. Defaults to "#1e1e1e"}

\item{line_color}{Line color for the plot. Defaults to "#e0e0e0"}

\item{title_color}{Text color for the title. Defaults to "white"}

\item{subtitle_color}{Text color the the subtitle. Defaults to "#8e8e93"}

\item{caption_color}{Text color the the caption. Defaults to be equal to the subtitle}

\item{tier_label_color}{Text color for the tier labels. Defaults to be equal to the title}

\item{na_headshot_to_logo}{Should NA/non-matches return the MLB logo instead of a grayed out blank headshot? Defaults to \code{TRUE}}
}
\value{
A plot object created with \code{\link[ggplot2:ggplot]{ggplot2::ggplot()}}.
}
\description{
This function sets up a ggplot to visualize MLB player tiers.
Adapted from \href{https://nflplotr.nflverse.com/reference/nfl_team_tiers.html}{\code{nflplotR}}
}
\examples{
\donttest{
library(ggplot2)
library(dplyr, warn.conflicts = FALSE)
player_ids <- load_headshots() |>
  head(35) |>
  pull(savant_id)

# Build the player tiers data frame
# This is completely random!
df <- data.frame(
  tier_no = sample(1:5, length(player_ids), replace = TRUE),
  player_id = player_ids
) \%>\%
  dplyr::group_by(tier_no) \%>\%
  dplyr::mutate(tier_rank = sample(1:n(), n()))

# Plot player tiers
mlb_player_tiers(df)

# Create a combined tier which is useful for tiers with lots of players that
# should be split up in two or more rows. This is done by setting an empty
# string for the tier 5 description and removing the tier separation line
# below tier number 4.
# This example also shows how to turn off the subtitle and add a caption
mlb_player_tiers(df,
                 subtitle = NULL,
                 caption = "This is the caption",
                 tier_desc = c("1" = "MVP Candidates",
                               "2" = "Very Good",
                               "3" = "Medium",
                               "4" = "A Combined Tier",
                               "5" = ""),
                 no_line_below_tier = 4)

# For the development of the tiers, it can be useful to turn off image
# rendering as this can take quite a long time. By setting `devel = TRUE`, the
# headshots are replaced by player ids which is much faster
mlb_player_tiers(df,
                 tier_desc = c("1" = "MVP Candidates",
                               "2" = "Very Good",
                               "3" = "",
                               "4" = "A Combined Tier",
                               "5" = ""),
                 no_line_below_tier = c(2, 4),
                 devel = TRUE)
}
}
