#' @title Print a `model_set`-Class Object
#'
#' @description Print the content of
#' a `model_set`-class object.
#'
#' @details
#' It is the print method of the
#' output of [model_set()].
#'
#' @return
#' `x` is returned invisibly.
#' Called for its side effect.
#'
#' @param x A `model_set`-class object.
#'
#' @param bic_digits The number of
#' decimal places to be displayed
#' for BIC.
#' Default is 3.
#'
#' @param bpp_digits The number of
#' decimal places to be displayed
#' for BIC posterior probability
#' and prior probabilities.
#' Default is 3.
#'
#' @param sort_models Whether the models
#' will be sorted by BIC posterior
#' probability.
#' Default is `TRUE`.
#'
#' @param max_models The maximum number
#' of models to be printed. Default is
#' 20.
#'
#' @param bpp_target The desired
#' BIC probability. Used to compute
#' and print
#' the minimum prior probability
#' of the target model required to
#' achieve `bpp_target`. Default
#' is `NULL`.
#'
#' @param ...  Optional arguments.
#' Ignored.
#'
#' @author Shu Fai Cheung <https://orcid.org/0000-0002-9871-9448>
#'
#' @seealso A `model_set`-class object
#' is generated by [model_set()].
#'
#' @examples
#'
#' library(lavaan)
#'
#' dat <- dat_path_model
#'
#' mod <-
#' "
#' x3 ~ a*x1 + b*x2
#' x4 ~ a*x1
#' ab := a*b
#' "
#'
#' fit <- sem(mod, dat_path_model, fixed.x = TRUE)
#'
#' out <- model_set(fit)
#' out
#'
#' @export

print.model_set <- function(x,
                            bic_digits = 3,
                            bpp_digits = 3,
                            sort_models = TRUE,
                            max_models = 20,
                            bpp_target = NULL,
                            ...) {
    fit_n <- length(x$models)
    fit_names <- names(x$models)
    models_fitted <- !is.null(x$fit)
    model_set_call <- x$model_set_call
    if (!models_fitted) {
        fit_many_call <- NULL
        k_converged <- NA
        k_post_check <- NA
      } else {
        fit_many_call <- x$call
        k_converged <- sum(sapply(x$converged, isTRUE))
        k_post_check <- sum(sapply(x$post_check, isTRUE))
      }
    if (!models_fitted) {
        change_tmp <- rep(NA, fit_n)
        prior_tmp <- rep(NA, fit_n)
        bic_tmp <- rep(NA, fit_n)
        postprob_tmp <- rep(NA, fit_n)
      } else {
        change_tmp <- x$change
        prior_tmp <- x$prior
        bic_tmp <- x$bic
        postprob_tmp <- x$bpp
      }
    out_table <- data.frame(modification = fit_names,
                            df = change_tmp,
                            Prior = prior_tmp,
                            BIC = bic_tmp,
                            BPP = postprob_tmp)
    if (sort_models && models_fitted) {
        i <- order(out_table$BPP,
                   decreasing = TRUE)
        out_table <- out_table[i, ]
        tmp <- round(cumsum(out_table$BPP), bpp_digits)
        tmp <- formatC(tmp,
                       digits = bpp_digits,
                       format = "f")
        out_table["Cumulative"] <- tmp
      }
    out_table_print <- out_table
    out_table_print$Prior <- round(out_table_print$Prior,
                                 digits = bpp_digits)
    out_table_print$Prior <- formatC(out_table_print$Prior,
                                   digits = bpp_digits,
                                   format = "f")
    out_table_print$BIC <- round(out_table_print$BIC,
                                 digits = bic_digits)
    out_table_print$BIC <- formatC(out_table_print$BIC,
                                   digits = bic_digits,
                                   format = "f")
    out_table_print$BPP <- round(out_table_print$BPP,
                                 digits = bpp_digits)
    out_table_print$BPP <- formatC(out_table_print$BPP,
                                   digits = bpp_digits,
                                   format = "f")
    if (isTRUE(fit_n > max_models)) {
        gt_max_models <- TRUE
        x_tmp <- out_table_print[seq_len(max_models), ]
      } else {
        gt_max_models <- FALSE
        x_tmp <- out_table_print
      }
    cat("\n")
    cat("Call:\n")
    print(model_set_call)
    cat("\n")
    if (models_fitted) {
        tmp1 <- c("Number of model(s) fitted",
                  "Number of model(s) converged",
                  "Number of model(s) passed post.check")
        tmp2 <- c(fit_n,
                  k_converged,
                  k_post_check)
        tmp3 <- auto_tab(tmp1,
                         tmp2,
                         between = ": ")
        cat(tmp3, sep = "\n")
        if (!is.null(bpp_target)) {
            bpp_min <- min_prior(x$bic,
                                 bpp_target = bpp_target,
                                 target_name = "original")
            tmp <- data.frame(x = c(
                      formatC(bpp_target, digits = bpp_digits, format = "f"),
                      formatC(bpp_min, digits = bpp_digits, format = "f"),
                      formatC(x$bpp["original"], digits = bpp_digits, format = "f")))
            colnames(tmp) <- "Target Model"
            rownames(tmp) <- c("Desired minimum BIC posterior probability:",
                               "Required minimum prior probability:",
                               "Current BIC posterior probability:")
            print(tmp)
          }
      } else {
        cat("Models are not fitted.")
        cat("\n")
      }
    cat("\n")
    tmp1 <- ifelse(sort_models,
                   " (sorted by BPP)",
                   "")
    if (!models_fitted) tmp1 <- ""
    if (gt_max_models) {
        cat("The first ",
            max_models,
            " model(s)",
            tmp1,
            ":\n", sep = "")
      } else {
        cat("The models",
            tmp1,
            ":\n", sep = "")
      }
    rownames(x_tmp) <- x_tmp$modification
    x_tmp$modification <- NULL
    print(x_tmp)
    cat("\nNote:\n")
    cat("- BIC: Bayesian Information Criterion.\n")
    cat("- BPP: BIC posterior probability.\n")
    if (sort_models) {
        cat("- Cumulative: Cumulative BIC posterior probability.\n")
      }
    if (gt_max_models) {
        x <- paste(fit_n,
                   "models were fitted but",
                   max_models,
                   "were printed. Call print() and",
                   "set 'max_models' to a larger number",
                   "to print more models, or set it to",
                   "NA to print all models.")
        catwrap(x, initial = "- ", exdent = 2)
      }
    invisible(x)
  }