% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelsummary.R
\name{modelsummary}
\alias{modelsummary}
\title{Model Summary Tables}
\usage{
modelsummary(
  models,
  output = "default",
  fmt = 3,
  estimate = "estimate",
  statistic = "std.error",
  vcov = NULL,
  conf_level = 0.95,
  exponentiate = FALSE,
  stars = FALSE,
  coef_map = NULL,
  coef_omit = NULL,
  coef_rename = NULL,
  gof_map = NULL,
  gof_omit = NULL,
  group = term ~ model,
  group_map = NULL,
  add_rows = NULL,
  align = NULL,
  notes = NULL,
  title = NULL,
  escape = TRUE,
  ...
)
}
\arguments{
\item{models}{a model or (optionally named) list of models}

\item{output}{filename or object type (character string)
\itemize{
\item Supported filename extensions: .docx, .html, .tex, .md, .txt, .png, .jpg.
\item Supported object types: "default", "html", "markdown", "latex", "latex_tabular", "data.frame", "gt", "kableExtra", "huxtable", "flextable", "jupyter". The "modelsummary_list" value produces a lightweight object which can be saved and fed back to the \code{modelsummary} function.
\item Warning: Users should not supply a file name to the \code{output} argument if they intend to customize the table with external packages. See the 'Details' section.
\item LaTeX compilation requires the \code{booktabs} and \code{siunitx} packages, but \code{siunitx} can be disabled or replaced with global options. See the 'Details' section.
\item The default output formats and table-making packages can be modified with global options. See the 'Details' section.
}}

\item{fmt}{determines how to format numeric values
\itemize{
\item integer: the number of digits to keep after the period \code{format(round(x, fmt), nsmall=fmt)}
\item character: passed to the \code{sprintf} function (e.g., '\%.3f' keeps 3 digits with trailing zero). See \code{?sprintf}
\item function: returns a formatted character string.
\item LaTeX output: To ensure proper typography, all numeric entries are enclosed in the \verb{\\num\{\}} command, which requires the \code{siunitx} package to be loaded in the LaTeX preamble. This behavior can be altered with global options. See the 'Details' section.
}}

\item{estimate}{string or \code{glue} string of the estimate to display (or a
vector with one string per model). Valid entries include any column name of
the data.frame produced by \code{get_estimates(model)}. Examples:
\itemize{
\item \code{"estimate"}
\item \code{"{estimate} ({std.error}){stars}"}
\item \code{"{estimate} [{conf.low}, {conf.high}]"}
}}

\item{statistic}{vector of strings or \code{glue} strings which select uncertainty
statistics to report vertically below the estimate. NULL omits all
uncertainty statistics.
\itemize{
\item "conf.int", "std.error", "statistic", "p.value", "conf.low", "conf.high",
or any column name produced by: \code{get_estimates(model)}
\item \code{glue} package strings with braces, such as:
\itemize{
\item \code{"{p.value} [{conf.low}, {conf.high}]"}
\item \code{"Std.Error: {std.error}"}
}
\item Parentheses are added automatically unless the string includes \code{glue} curly braces \code{{}}.
}}

\item{vcov}{robust standard errors and other manual statistics. The \code{vcov}
argument accepts six types of input (see the 'Details' and 'Examples'
sections below):
\itemize{
\item NULL returns the default uncertainty estimates of the model object
\item string, vector, or (named) list of strings. "iid", "classical", and "constant" are aliases for \code{NULL}, which returns the model's default uncertainty estimates. The strings "HC", "HC0", "HC1" (alias: "stata"), "HC2", "HC3" (alias: "robust"), "HC4", "HC4m", "HC5", "HAC", "NeweyWest", "Andrews", "panel-corrected", "outer-product", and "weave" use variance-covariance matrices computed using functions from the \code{sandwich} package, or equivalent method. The behavior of those functions can (and sometimes \emph{must}) be altered by passing arguments to \code{sandwich} directly from \code{modelsummary} through the ellipsis (\code{...}), but it is safer to define your own custom functions as described in the next bullet.
\item function or (named) list of functions which return variance-covariance matrices with row and column names equal to the names of your coefficient estimates (e.g., \code{stats::vcov}, \code{sandwich::vcovHC}, \code{function(x) vcovPC(x, cluster="country")}).
\item formula or (named) list of formulas with the cluster variable(s) on the right-hand side (e.g., ~clusterid).
\item named list of \code{length(models)} variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
\item a named list of length(models) vectors with names equal to the names of your coefficient estimates. See 'Examples' section below. Warning: since this list of vectors can include arbitrary strings or numbers, \code{modelsummary} cannot automatically calculate p values. The \code{stars} argument may thus use incorrect significance thresholds when \code{vcov} is a list of vectors.
}}

\item{conf_level}{confidence level to use for confidence intervals}

\item{exponentiate}{TRUE, FALSE, or logical vector of length equal to the
number of models. If TRUE, the \code{estimate}, \code{conf.low}, and \code{conf.high}
statistics are exponentiated, and the \code{std.error} is transformed to
\code{exp(estimate)*std.error}.}

\item{stars}{to indicate statistical significance
\itemize{
\item FALSE (default): no significance stars.
\item TRUE: +=.1, *=.05, **=.01, ***=0.001
\item Named numeric vector for custom stars such as \code{c('*' = .1, '+' = .05)}
\item Note: a legend will not be inserted at the bottom of the table when the \code{estimate} or \code{statistic} arguments use "glue strings" with \code{{stars}}.
}}

\item{coef_map}{character vector. Subset, rename, and reorder coefficients.
Coefficients omitted from this vector are omitted from the table. The order
of the vector determines the order of the table.  \code{coef_map} can be a named
or an unnamed character vector (see the Examples section below). If
\code{coef_map} is a named vector, its values define the labels that must appear
in the table, and its names identify the original term names stored in the
model object: \code{c("hp:mpg"="HPxM/G")}.}

\item{coef_omit}{string regular expression. Omits all matching coefficients
from the table using \code{grepl(perl=TRUE)}. This argument uses perl-compatible regular expressions, which allows expressions such as \verb{"Int|ABC" which omits coefficients matching either "Int" or "ABC", and }"^(?!.*Intercept)"` which omits every term except the intercept.}

\item{coef_rename}{named character vector or function which returns a named
vector. Values of the vector refer to the variable names that will appear
in the table. Names refer to the original term names stored in the model
object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.
If you provide a function to \code{coef_rename}, \code{modelsummary} will create a named
vector for you by deriving the new variable names from the vector of original
term names with your function.}

\item{gof_map}{rename, reorder, and omit goodness-of-fit statistics and other
model information. This argument accepts 4 types of values:
\itemize{
\item NULL (default): the \code{modelsummary::gof_map} dictionary is used for formatting, and all unknown statistic are included.
\item character vector such as \code{c("rmse", "nobs", "r.squared")}. Elements correspond to colnames in the data.frame produced by \code{get_gof(model)}. The default dictionary is used to format and rename statistics.
\item data.frame with 3 columns named "raw", "clean", "fmt". Unknown statistics are omitted. See the 'Examples' section below.
\item list of lists, each of which includes 3 elements named "raw", "clean", "fmt". Unknown statistics are omitted. See the 'Examples section below'.
}}

\item{gof_omit}{string regular expression. Omits all matching gof statistics from
the table. This argument uses perl-compatible regular expressions (\code{grepl(perl=TRUE)}), which allows expressions such as \code{".*"} which omits everything, and \code{"^(?!R2|Num)"} which omits every term except those that start with "R2" or "Num".}

\item{group}{transpose or group models and estimates (formula). The left
side of the formula represents rows and the right side columns.
\itemize{
\item Formula with two components called \code{term} and \code{model}.
\itemize{
\item \code{term ~ model}
\item \code{model ~ term}
}
\item Formula with three components, including one of the column names in the data frame produced by \code{get_estimates(model)}. Examples:
\itemize{
\item \code{response + term ~ model}
\item \code{term ~ model + y.level}
}
}}

\item{group_map}{named or unnamed character vector. Subset, rename, and
reorder coefficient groups specified in the \code{group} argument. See \code{coef_map}.}

\item{add_rows}{a data.frame (or tibble) with the same number of columns as
your main table. By default, rows are appended to the bottom of the table.
You can define a "position" attribute of integers to set the row positions.
See Examples section below.}

\item{align}{A string with a number of characters equal to the number of columns in
the table (e.g., \code{align = "lcc"}).  Valid characters: l, c, r, d.
\itemize{
\item "l": left-aligned column
\item "c": centered column
\item "r": right-aligned column
\item "d": dot-aligned column. Only supported for LaTeX/PDF tables produced by \code{kableExtra}. These commands must appear in the LaTeX preamble (they are added automatically when compiling Rmarkdown documents to PDF):
\itemize{
\item \verb{\\usepackage\{booktabs\}}
\item \verb{\\usepackage\{siunitx\}}
\item \verb{\\newcolumntype\{d\}\{S[input-symbols = ()]\}}
}
}}

\item{notes}{list or vector of notes to append to the bottom of the table.}

\item{title}{string}

\item{escape}{boolean TRUE escapes or substitutes LaTeX/HTML characters which could
prevent the file from compiling/displaying. This setting does not affect captions or notes.}

\item{...}{all other arguments are passed through to the extractor and
table-making functions (by default \code{broom::tidy} and \code{kableExtra::kbl}, but
this can be customized). This allows users to pass arguments directly to
\code{modelsummary} in order to affect the behavior of other functions behind the
scenes. For example,
\itemize{
\item \code{performance::model_performance(metrics="RMSE")} to select goodness-of-fit statistics to extract using the \code{performance} package (must have set \code{options(modelsummary_get="easystats")} first). This can be useful for some models when statistics take a long time to compute.
}}
}
\value{
a regression table in a format determined by the \code{output} argument.
}
\description{
Create beautiful and customizable tables to summarize several statistical
models side-by-side. This function supports dozens of statistical models,
and it can produce tables in HTML, LaTeX, Word, Markdown, PDF, PowerPoint,
Excel, RTF, JPG, or PNG. The appearance of the tables can be customized
extensively by specifying the \code{output} argument, and by using functions from
one of the supported table customization packages: \code{kableExtra}, \code{gt},
\code{flextable}, \code{huxtable}. This function's behavior can be altered using
global options (See the \emph{Details} section below.)
}
\details{
\subsection{\code{output}}{

The \code{modelsummary_list} output is a lightweight format which can be used to save model results, so they can be fed back to \code{modelsummary} later to avoid extracting results again.

When a file name with a valid extension is supplied to the \code{output} argument,
the table is written immediately to file. If you want to customize your table
by post-processing it with an external package, you need to choose a
different output format and saving mechanism. Unfortunately, the approach
differs from package to package:
\itemize{
\item \code{gt}: set \code{output="gt"}, post-process your table, and use the \code{gt::gtsave} function.
\item \code{kableExtra}: set \code{output} to your destination format (e.g., "latex", "html", "markdown"), post-process your table, and use \code{kableExtra::save_kable} function.
}
}

\subsection{\code{vcov}}{

To use a string such as "robust" or "HC0", your model must be supported
by the \code{sandwich} package. This includes objects such as: lm, glm,
survreg, coxph, mlogit, polr, hurdle, zeroinfl, and more.

NULL, "classical", "iid", and "constant" are aliases which do not modify
uncertainty estimates and simply report the default standard errors stored
in the model object.

One-sided formulas such as \code{~clusterid} are passed to the \code{sandwich::vcovCL}
function.

Matrices and functions producing variance-covariance matrices are first
passed to \code{lmtest}. If this does not work, \code{modelsummary} attempts to take
the square root of the diagonal to adjust "std.error", but the other
uncertainty estimates are not be adjusted.

Numeric vectors are formatted according to \code{fmt} and placed in brackets.
Character vectors printed as given, without parentheses.

If your model type is supported by the \code{lmtest} package, the
\code{vcov} argument will try to use that package to adjust all the
uncertainty estimates, including "std.error", "statistic", "p.value", and
"conf.int". If your model is not supported by \code{lmtest}, only the "std.error"
will be adjusted by, for example, taking the square root of the matrix's
diagonal.
}
}
\section{Global Options}{


The behavior of \code{modelsummary} can be affected by setting global options:
\itemize{
\item \code{modelsummary_factory_default}
\item \code{modelsummary_factory_latex}
\item \code{modelsummary_factory_html}
\item \code{modelsummary_factory_png}
\item \code{modelsummary_get}
\item \code{modelsummary_format_numeric_latex}
\item \code{modelsummary_format_numeric_html}
}
\subsection{Table-making packages}{

\code{modelsummary} supports 4 table-making packages: \code{kableExtra}, \code{gt},
\code{flextable}, and \code{huxtable}. Some of these packages have overlapping
functionalities. For example, 3 of those packages can export to LaTeX. To
change the default backend used for a specific file format, you can use
the \code{options} function:

\code{options(modelsummary_factory_html = 'kableExtra')}
\code{options(modelsummary_factory_latex = 'gt')}
\code{options(modelsummary_factory_word = 'huxtable')}
\code{options(modelsummary_factory_png = 'gt')}
}

\subsection{Model extraction functions}{

\code{modelsummary} can use two sets of packages to extract information from
statistical models: \code{broom} and the \code{easystats} family (\code{performance} and
\code{parameters}). By default, it uses \code{broom} first and \code{easystats} as a
fallback if \code{broom} fails. You can change the order of priorities
or include goodness-of-fit extracted by \emph{both} packages by setting:

\code{options(modelsummary_get = "broom")}
\code{options(modelsummary_get = "easystats")}
\code{options(modelsummary_get = "all")}
}

\subsection{Formatting numeric entries}{

By default, LaTeX tables enclose all numeric entries in the \verb{\\num\{\}} command
from the siunitx package. To prevent this behavior, or to enclose numbers
in dollar signs (for LaTeX math mode), users can call:

\code{options(modelsummary_format_numeric_latex = "plain")}
\code{options(modelsummary_format_numeric_latex = "mathmode")}

A similar option can be used to display numerical entries using MathJax in
HTML tables:

\code{options(modelsummary_format_numeric_html = "mathjax")}
}
}

\examples{
\dontrun{

# The `modelsummary` website includes \emph{many} examples and tutorials:
# https://vincentarelbundock.github.io/modelsummary

library(modelsummary)

# load data and estimate models
data(trees)
models <- list()
models[['Bivariate']] <- lm(Girth ~ Height, data = trees)
models[['Multivariate']] <- lm(Girth ~ Height + Volume, data = trees)

# simple table
modelsummary(models)

# statistic
modelsummary(models, statistic = NULL)
modelsummary(models, statistic = 'p.value')
modelsummary(models, statistic = 'statistic')
modelsummary(models, statistic = 'conf.int', conf_level = 0.99)
modelsummary(models, statistic = c("t = {statistic}",
                                   "se = {std.error}",
                                   "conf.int"))

# estimate
modelsummary(models,
  statistic = NULL,
  estimate = "{estimate} [{conf.low}, {conf.high}]")
modelsummary(models,
  estimate = c("{estimate}{stars}",
               "{estimate} ({std.error})"))

# vcov
modelsummary(models, vcov = "robust")
modelsummary(models, vcov = list("classical", "stata"))
modelsummary(models, vcov = sandwich::vcovHC)
modelsummary(models,
  vcov = list(stats::vcov, sandwich::vcovHC))
modelsummary(models,
  vcov = list(c("(Intercept)"="", "Height"="!"),
              c("(Intercept)"="", "Height"="!", "Volume"="!!")))

# vcov with custom names
modelsummary(
  models,
  vcov = list("Stata Corp" = "stata",
              "Newey Lewis & the News" = "NeweyWest"))

# coef_rename
modelsummary(models, coef_rename = c('Volume' = 'Large', 'Height' = 'Tall'))
modelsummary(models, coef_rename = toupper)
modelsummary(models, coef_rename = coef_rename)

# coef_map
modelsummary(models, coef_map = c('Volume' = 'Large', 'Height' = 'Tall'))
modelsummary(models, coef_map = c('Volume', 'Height'))

# title
modelsummary(models, title = 'This is the title')

# title with LaTeX label (for numbering and referencing)
modelsummary(models, title = 'This is the title \\\\label{tab:description}')

# add_rows
rows <- tibble::tribble(~term, ~Bivariate, ~Multivariate,
  'Empty row', '-', '-',
  'Another empty row', '?', '?')
attr(rows, 'position') <- c(1, 3)
modelsummary(models, add_rows = rows)

# notes
modelsummary(models, notes = list('A first note', 'A second note'))

# gof_map: tribble
library(tibble)
gm <- tribble(
  ~raw,        ~clean,      ~fmt,
  "r.squared", "R Squared", 5)
modelsummary(models, gof_map = gm)

# gof_map: data.frame
gm <- modelsummary::gof_map
gof_custom$omit[gof_custom$raw == 'deviance'] <- FALSE
gof_custom$fmt[gof_custom$raw == 'r.squared'] <- "\%.5f"
modelsummary(models, gof_map = gof_custom)

# gof_map: list of lists
f1 <- function(x) format(round(x, 3), big.mark=",")
f2 <- function(x) format(round(x, 0), big.mark=",")
gm <- list(
  list("raw" = "nobs", "clean" = "N", "fmt" = f2),
  list("raw" = "AIC", "clean" = "aic", "fmt" = f1))
modelsummary(models,
  fmt = f1,
  gof_map = gm)

}

}
