% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modeltime-forecast.R
\name{modeltime_forecast}
\alias{modeltime_forecast}
\title{Forecast future data}
\usage{
modeltime_forecast(
  object,
  new_data = NULL,
  h = NULL,
  actual_data = NULL,
  conf_interval = 0.8,
  ...
)
}
\arguments{
\item{object}{A Modeltime Table that has been calibrated with \code{\link[=modeltime_calibrate]{modeltime_calibrate()}}}

\item{new_data}{A \code{tibble} containing future information to forecast.
If \code{NULL}, forecasts the calibration data.}

\item{h}{The forecast horizon (can be used instead of \code{new_data} for
time series with no exogenous regressors).
Extends the calibration data \code{h} periods into the future.}

\item{actual_data}{Reference data that is combined with the output tibble and given a \code{.key = "actual"}}

\item{conf_interval}{An estimated confidence interval based on the in-sample residuals}

\item{...}{Not currently used}
}
\value{
A tibble with predictions and time-stamp data. For ease of plotting and calculations,
the column names are transformed to:
\itemize{
\item \code{.key}: Values labeled either "prediction" or "actual"
\item \code{.index}: The timestamp index.
\item \code{.value}: The value being forecasted.
\item \code{.conf_lo}: The lower limit of the confidence interval.
\item \code{.conf_hi}: The upper limit of the confidence interval.
}

Additional descriptive columns are included:
\itemize{
\item \code{.model_id}: Model ID from the Modeltime Table
\item \code{.model_desc}: Model Description from the Modeltime Table
}

Unnecessary columns are \emph{dropped} to save space:
\itemize{
\item \code{.model}
\item \code{.calibration_data}
}
}
\description{
The goal of \code{modeltime_forecast()} is to simplify the process of
forecasting future data.
}
\details{
The key parameters are (controlled by \code{new_data} or \code{h}) and
combining with existing data (controlled by \code{actual_data}) in preparation
for visualization with \code{\link[=plot_modeltime_forecast]{plot_modeltime_forecast()}}.

\strong{Specifying New Data or Horizon (h)}

When forecasting you can specify future data using:
\enumerate{
\item \code{new_data}: This is a future tibble with date column and columns for xregs
extending the trained dates and exogonous regressors (xregs) if used.
\itemize{
\item \strong{Forecasting Evaluation Data}: By default, the \code{new_data} will use the \code{.calibration_data}
if \code{new_data} is not provided.
This is the equivalent of using \code{rsample::testing()} for getting test data sets.
\item \strong{Forecasting Future Data}: See \code{\link[=future_frame]{future_frame()}} for creating future tibbles.
\item \strong{Xregs}: Can be used with this method
}
\item \code{h}: This is a phrase like "1 year", which extends the \code{.calibration_data} into the future.
\itemize{
\item \strong{Forecasting Future Data}: All forecasts using \code{h} are
\emph{\strong{extended after the calibration data}}, which is
desirable \emph{after refitting} with \code{\link[=modeltime_refit]{modeltime_refit()}}.
Internally, a call is made to \code{\link[=future_frame]{future_frame()}} to
expedite creating new data using the date feature.
\item \strong{Xregs}: Cannot be used because future data must include new xregs.
}
}

\strong{Actual Data}

This is reference data that contains the true values of the time-stamp data.
It helps in visualizing the performance of the forecast vs the actual data.

\strong{Confidence Interval Estimation}

Confidence intervals (\code{.conf_lo}, \code{.conf_hi}) are estimated based on the normal estimation of
the testing errors (out of sample) from \code{\link[=modeltime_calibrate]{modeltime_calibrate()}}.
The out-of-sample error estimates are then carried through and
applied to applied to any future forecasts.

The confidence interval can be adjusted with the \code{conf_interval} parameter. An
80\% confidence interval estimates a normal (Gaussian distribution) that assumes that
80\% of the future data will fall within the upper and lower confidence limits.

The confidence interval is \emph{mean-adjusted}, meaning that if the mean of the residuals
is non-zero, the confidence interval is adjusted to widen the interval to capture
the difference in means.

Refitting has no affect on the confidence interval since this is calculated independently of
the refitted model (on data with a smaller sample size). New observations typically improve
future accuracy, which in most cases makes the out-of-sample confidence intervals conservative.
}
\examples{
library(tidyverse)
library(lubridate)
library(timetk)
library(parsnip)
library(rsample)

# Data
m750 <- m4_monthly \%>\% filter(id == "M750")

# Split Data 80/20
splits <- initial_time_split(m750, prop = 0.9)

# --- MODELS ---

# Model 1: auto_arima ----
model_fit_arima <- arima_reg() \%>\%
    set_engine(engine = "auto_arima") \%>\%
    fit(value ~ date, data = training(splits))


# ---- MODELTIME TABLE ----

models_tbl <- modeltime_table(
    model_fit_arima
)

# ---- CALIBRATE ----

calibration_tbl <- models_tbl \%>\%
    modeltime_calibrate(new_data = testing(splits))

# ---- ACCURACY ----

calibration_tbl \%>\%
    modeltime_accuracy()

# ---- FORECAST ----

calibration_tbl \%>\%
    modeltime_forecast(
        new_data    = testing(splits),
        actual_data = m750
    )

}
