% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lsp_compare.R
\name{lsp_compare}
\alias{lsp_compare}
\title{Comparison between spatial patterns}
\usage{
lsp_compare(
  x,
  y,
  type,
  dist_fun,
  window = NULL,
  output = "stars",
  neighbourhood = 4,
  threshold = 0.5,
  ordered = FALSE,
  repeated = FALSE,
  normalization = "pdf",
  wecoma_fun = "mean",
  wecoma_na_action = "replace",
  ...
)
}
\arguments{
\item{x}{Object of class \code{stars}, \code{stars_proxy}, or terra's \code{SpatRaster}. It should have one attribute (for \code{"coma"}, \code{"cove"}), two attributes (\code{"cocoma"}, \code{"cocove"}, \code{"wecoma"}, \code{"wecove"}), two or more attributes (\code{"incoma"}, \code{"incove"}), or any number of attributes suitable for user-defined functions.}

\item{y}{Object of class \code{stars}, \code{stars_proxy}, or terra's \code{SpatRaster}. It should have one attribute (for \code{"coma"}, \code{"cove"}), two attributes (\code{"cocoma"}, \code{"cocove"}, \code{"wecoma"}, \code{"wecove"}), two or more attributes (\code{"incoma"}, \code{"incove"}), or any number of attributes suitable for user-defined functions.}

\item{type}{Type of the calculated signature. It can be \code{"coma"} (co-occurrence matrix), \code{"cove"} (co-occurrence vector), \code{"cocoma"} (co-located co-occurrence matrix), \code{"cocove"} (co-located co-occurrence vector), \code{"wecoma"} (weighted co-occurrence matrix), \code{"wecove"} (weighted co-occurrence vector), \code{"incoma"} (integrated co-occurrence matrix), \code{"incove"} (integrated co-occurrence vector), \code{"composition"} or any function that can summarize \code{stars} objects.}

\item{dist_fun}{Distance measure used. This function uses the \code{philentropy::distance} function in the background. Run \code{philentropy::getDistMethods()} to find possible distance measures.}

\item{window}{Specifies areas for analysis. It can be either: \code{NULL}, a numeric value, or an \code{sf} object. If \code{window=NULL} calculations are performed for a whole area. If the \code{window} argument is numeric, it is a length of the side of a square-shaped block of cells. Expressed in the numbers of cells, it defines the extent of a local pattern. If an \code{sf} object is provided, each feature (row) defines the extent of a local pattern. The \code{sf} object should have one attribute (otherwise, the first attribute is used as an id).}

\item{output}{The class of the output. Either \code{"stars"}, \code{"sf"}, or \code{"terra"}}

\item{neighbourhood}{The number of directions in which cell adjacencies are considered as neighbours:
4 (rook's case) or 8 (queen's case). The default is 4.}

\item{threshold}{The share of NA cells to allow metrics calculation.}

\item{ordered}{For \code{"cove"}, \code{"cocove"}, \code{"wecove"} and \code{"incove"} only. The type of pairs considered.
Either "ordered" (TRUE) or "unordered" (FALSE).
The default is FALSE.}

\item{repeated}{For \code{"incove"} only. Should the repeated co-located co-occurrence matrices be used?
Either "ordered" (TRUE) or "unordered" (FALSE).
The default is FALSE.}

\item{normalization}{For \code{"cove"}, \code{"cocove"}, \code{"wecove"}, \code{"incove"}, \code{"composition"}, or user-provided functions only.
Should the output vector be normalized?
Either "none" or "pdf".
The "pdf" option normalizes a vector to sum to one.
The default is "pdf".}

\item{wecoma_fun}{For \code{"wecoma"} and \code{"wecove"} only. Function to calculate values from adjacent cells to contribute to exposure matrix, \code{"mean"} - calculate average values of local population densities from adjacent cells, \code{"geometric_mean"} - calculate geometric mean values of local population densities from adjacent cells, or \code{"focal"} assign a value from the focal cell}

\item{wecoma_na_action}{For \code{"wecoma"} and \code{"wecove"} only. Decides on how to behave in the presence of missing values in \code{w}. Possible options are \code{"replace"}, \code{"omit"}, \code{"keep"}. The default, \code{"replace"}, replaces missing values with 0, \code{"omit"} does not use cells with missing values, and \code{"keep"} keeps missing values.}

\item{...}{Additional arguments for the \code{philentropy::distance} function.}
}
\value{
Object of class \code{stars} (or \code{sf} or terra's \code{SpatRaster}, depending on the \code{output} argument).
It has four attributes:
(1) \code{id} - an id of each window.
For irregular windows, it is the values provided in the \code{window} argument,
(2) \code{na_prop_x} - share (0-1) of \code{NA} cells for each window in the \code{x} object,
(3) \code{na_prop_y} - share (0-1) of \code{NA} cells for each window in the \code{y} object,
(4) \code{dist}- calculated distance between signatures for each window
}
\description{
Compares two spatial datasets containing categorical raster data.
It accepts a categorical raster dataset with one or more attributes, and compares it to the second dataset with the same attributes and dimensions.
The both dataset are either compared to as whole areas, areas divided into regular windows, or areas divided into irregular windows.
This function allows for several types of comparisons using different representations of spatial patterns, including "coma" (co-occurrence matrix), "cove" (co-occurrence vector), "cocoma" (co-located co-occurrence matrix), "cocove" (co-located co-occurrence vector), "wecoma" (weighted co-occurrence matrix), "wecove" (weighted co-occurrence vector), "incoma" (integrated co-occurrence matrix), "incove" (integrated co-occurrence vector). These representations are created for both datasets, and next a distance between them is calculated using a selected measure from the \code{philentropy::distance} function.
Additional parameters, such as neighbourhood or normalization types, are also available.
}
\examples{
library(stars)

lc15 = read_stars(system.file("raster/landcover2015s.tif", package = "motif"))
lc01 = read_stars(system.file("raster/landcover2001s.tif", package = "motif"))
ecoregions = read_sf(system.file("vector/ecoregionss.gpkg", package = "motif"))

ecoregions = st_transform(ecoregions, st_crs(lc15))

c1 = lsp_compare(lc01, lc15, type = "cove",
    dist_fun = "jensen-shannon", window = ecoregions["id"])
plot(c1["dist"])

\donttest{
# larger data example
library(stars)

lc15 = read_stars(system.file("raster/landcover2015.tif", package = "motif"))
lc01 = read_stars(system.file("raster/landcover2001.tif", package = "motif"))
ecoregions = read_sf(system.file("vector/ecoregions.gpkg", package = "motif"))

ecoregions = st_transform(ecoregions, st_crs(lc15))

c1 = lsp_compare(lc01, lc15, type = "cove",
    dist_fun = "jensen-shannon", window = ecoregions["id"])
plot(c1["dist"])
}
}
