% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mpitb.est.R
\name{mpitb.est}
\alias{mpitb.est}
\alias{mpitb.est.mpitb_set}
\title{Estimate multidimensional poverty indices based on the Alkire-Foster method}
\usage{
mpitb.est(set, ...)

\method{mpitb.est}{mpitb_set}(
  set,
  klist = NULL,
  weights = "equal",
  measures = c("M0", "H", "A"),
  indmeasures = c("hd", "hdk", "actb", "pctb"),
  indklist = NULL,
  over = NULL,
  ...,
  cotyear = NULL,
  tvar = NULL,
  cotmeasures = c("M0", "H", "A", "hd", "hdk"),
  ann = FALSE,
  cotklist = NULL,
  cotoptions = "total",
  noraw = FALSE,
  nooverall = FALSE,
  level = 0.95,
  multicore = getOption("mpitb.multicore"),
  verbose = TRUE
)
}
\arguments{
\item{set}{a "mpitb_set"-class object in which data, indicators, names and description have been specified.}

\item{...}{other arguments}

\item{klist}{a numeric vector representing the poverty cut-offs for calculating the MPI. Should be values between 1 and 100.}

\item{weights}{either a character value or a numeric vector. If "equal", it automatically calculated equal nested weights.}

\item{measures}{a character vector with the MPI and partial measures. Default include all the measures \code{c("M0","A","H")}. For more information, see Details section below.}

\item{indmeasures}{a character vector with the indicator-specific measures. Default include all the measures \code{c("hd", "hdk", "actb", "pctb")}. For more information, see Details section below.}

\item{indklist}{a numeric vector representing the poverty cut-offs for calculating indicator-specific measures. Should be values between 1 and 100. If \code{NULL}, it will be equal to \code{klist}.}

\item{over}{a character vector with columns names of the population subgroups in data.}

\item{cotyear}{a character value containing the column name of the years variable in the data. This argument is required if annualized changes over time measure are desired.}

\item{tvar}{a character value containing the column name of the time ID variable in the data.  This argument determines if changes over time are calculated.}

\item{cotmeasures}{a character vector with the changes over time measures. Default include all the measures \code{c("M0","A","H","hd", "hdk")}. For more information, see Details section below.}

\item{ann}{logical. If \code{TRUE}, annualized changes over time measure are estimated. If \code{FALSE}, only  non-annualized changes over time are calculated. Default value is \code{FALSE}.}

\item{cotklist}{a numeric vector representing the poverty cut-offs for calculating changes over time measures. Should be values between 1 and 100. If \code{NULL}, it will be equal to \code{klist}.}

\item{cotoptions}{a character vector. If "total", estimates change over the total period of observation, i.e. from the first year of observation to the last year of observation. If "insequence", then  estimates all consecutive (i.e. year-to-year) changes. The default is "total"}

\item{noraw}{logical. If \code{TRUE}, non-annualized changes over time measure are not estimated. Default is \code{FALSE}.}

\item{nooverall}{logical. If \code{TRUE}, estimations over all the observations are omitted, e.g., national level calculations, and only measure for the specified subgroups are estimated. Default is \code{FALSE}.}

\item{level}{numeric value with the desired confidence level for the confidence interval calculations in decimal format. Default value is 0.95.}

\item{multicore}{logical. Use \code{multicore} package for parallel estimation by measure and poverty cut-off over multiple processors? It uses forking approach. See Details below.}

\item{verbose}{logical. If TRUE, print information messages to the console so the user can control the dimension and indicators weights and which measures are being estimated in the process. If FALSE, these messages are ommited.}
}
\value{
An object with S3 class "mpitb_est" containing two data frames with the estimates of the cross-sectional measures ("lframe"-class) and changes over time ("cotframe"-class).
}
\description{
Estimate multidimensional poverty indices (MPI) based on the Alkire-Foster (AF) method including
disaggregated cross-sectional and changes over time estimates as well as quantities such as standard
errors and confidences intervals (accounting for the household survey design).
}
\details{
This functions is a S3 method for "mpitb_set" class. Hence, the project
has to be correctly specified with \code{mpitb.set()} function previously.

The vector of poverty cut-offs (\eqn{k}) in percentage point, i.e., numbers
between 1 and 100. Although the deprivation score (\eqn{c_i = \sum_{i=1}^n w_j g_{ij}^0}) is a real-valued
function, given the weights, it will assume a limited number of values. The same
occurs with the censored deprivation score. Therefore, despite accepting infinite
number of values, results may not vary with close values of \eqn{k}. For this reason,
it is recommended to use a very limited number of poverty cut-offs for the analysis.

If nothing is passed to \code{weights} argument, equal nested weights are calculated
by dimension and indicator. In this case, it is preferred to pass indicators as a list
in \code{mpitb.set()}. If the user wants to pass another weighting
scheme, she should first pass the indicators as a character vector in \code{mpitb.set()}
and then pass a numeric vector in \code{weights} such that the elements of this
vector match with the vector of indicators and all the weights sum up to 1.

To specify the population subgroups (e.g., living area, sex, etc.) and estimate the
disaggregated measures by each level of the subgroup, the user should pass the column
names of the population subgroups in the data using \code{over} argument. If \code{over}
is \code{NULL}, the measure are estimate using all the observations (e.g., national-level).
If population subgroups are specified and \code{nooverall} is set to \code{TRUE},
 aggregate (or national-level) estimates will not be produced.

Details on the AF measures estimation:

Available measures include the Adjusted Headcount Ratio (\eqn{M_0}), the
Incidence (\eqn{H}) and the Intensity of poverty (\eqn{A}), as well as other
indicator-specific measures
such as the uncensored headcount ratio (\eqn{h_j}), the censored headcount ratio (\eqn{h_j(k)})
and the absolute and percentage contribution.

The three first partial measures are pass in \code{measures} argument. By default,
\code{mpitb.est} calculates every measure \code{c("M0","H","A")}. The poverty
cut-off (\eqn{k}) for these measures estimation is specified in \code{klist} argument.

The indicator-specific measure are passed in \code{indmeasures} argument. By default,
\code{mpitb.est} calculates every measure \code{c("hd","hdk","actb","pctb")}. The poverty
cut-off (\eqn{k}) for these measures estimation is specified in \code{indklist} argument.
If \code{indklist} is \code{NULL}, poverty cut-offs in \code{klist} is used.
The absolute contribution \code{c("actb")} cannot be estimated without also
passing the censored headcount
ratios of each indicator  \code{c("hdk")} and the percentage contribution cannot be
calculated without \code{c("hdk")} and \code{c("M0")} passed in \code{measures} argument.

If any of these arguments is \code{NULL}, \code{mpitb.est()} skips these
measures. So it is useful for avoid calculating unnecessary estimations. For example,
if \code{measures = c("H","A")} and \code{indmeasures = NULL}, only the Incidence and
the Intensity will be estimated.

Details on changes over time measures:

The user can decide which AF measure changes over time she want to study. This is
set in \code{cotmeasures}. By default it calculates all the measure, except contributions, i.e.,
\code{cotmeasure = c("M0","A","H","hd","hdk")}. It would be important to check this argument in order to save time.
The poverty
cut-off (\eqn{k}) for these measures estimation is specified in \code{cotklist} argument.
If \code{cotklist} is \code{NULL}, poverty cut-offs in \code{klist} is used.
The standard errors of the changes over time measures is estimated using Delta method.

For calculating any point estimate for each time period and any change over time
measure, \code{tvar} should not be \code{NULL}. This argument should be a character with the
column name that references the time period \eqn{t = 1, \ldots,T}.

Changes over time measure can also be annualized. For such measure, information about
the years is needed. \code{cotyear}  should be a character with the column name
that have information about the years. Decimal digits are permitted. Argument \code{ann} is a logical
value. If \code{TRUE}, annualized measures are calculated. If \code{cotyear} is passed, \code{ann}
is automatically set to \code{TRUE}. If the former is not \code{NULL} and
\code{ann} is \code{FALSE}, only non-annualized
measures are estimated. If only annualized measure are
under study, the user can switch \code{noraw} to \code{TRUE} to avoid estimating non-annualized changes.

Finally, if there are more than two years survey rounds, the user can decide if estimate the
change over the total period of observation, i.e. from the first year of observation to the
last year of observation or year-to-year changes. To do the former, \code{cotoptions = "total"}
whereas for the latter case, \code{cotoptions = "insequence"}. By default, \code{cotoptions = "total"}
to avoid unnecessary estimations.

Some details on other arguments and estimations:

The package includes the possibility to do parallel calculations over all the measures and poverty cut-offs.
If \code{multicore} is \code{TRUE}, the package proceeds with parallel estimations. Caveat: this package uses
Forking method for parallelization which is only available on Unix-like systems (including Linux), i.e., Windows
users cannot benefit from parallelization.

For every measure the standard errors and confidence intervals are estimated. The former are estimated taking into
account the survey structure whereas the latter are estimated considering measures as proportions using \code{svyciprop()}
function from "survey" R package (it uses the "logit" method which consists of fitting a logistic regression model
and computes a Wald-type interval on the log-odds scale, which is then transformed to the probability scale).
}
\examples{
library(mpitbR)

data <- subset(syn_cdta)
data <- na.omit(data)

svydata <- survey::svydesign(id=~psu, weights = ~weight, strata = ~stratum, data = data)

indicators <- list(d1 = c("d_nutr","d_cm"),
                   d2 = c("d_satt","d_educ"),
                   d3 = c("d_elct","d_sani","d_wtr","d_hsg","d_ckfl","d_asst"))

# Specify mpitb project
set <- mpitb.set(svydata, indicators = indicators, name = "myname", desc = "pref. desc")

# Estimate the cross-sectional MPI and compare non-annualized changes over time
est <- mpitb.est(set, klist = c(33), measures = "M0", indmeasures = NULL,
                 tvar = "t", cotmeasures = "M0",
                 weights = "equal", over = c("area"))

coef(subset(est$lframe, measure == "M0" & t == 1))
confint(subset(est$lframe, measure == "M0" & t == 1))
summary(subset(est$lframe, measure == "M0" & t == 1))

coef(subset(est$cotframe, measure == "M0"))
confint(subset(est$cotframe, measure == "M0"))
summary(subset(est$cotframe, measure == "M0" & ctype == "abs" & ann == 0 & k == 33))
}
\references{
\emph{Alkire, S., Foster, J. E., Seth, S., Santos, M. E., Roche, J., & Ballon, P. (2015). Multidimensional poverty measurement and analysis. Oxford University Press.}

             \emph{Alkire, S., Roche, J. M., & Vaz, A. (2017). Changes over time in multidimensional poverty: Methodology and results for 34 countries. World Development, 94, 232-249}. \doi{10.1016/j.worlddev.2017.01.011}

             \emph{Suppa, N. (2023). mpitb: A toolbox for multidimensional poverty indices. The Stata Journal, 23(3), 625-657}. \doi{10.1177/1536867X231195286}
}
\seealso{
\code{coef}, \code{confint}, and \code{summary} methods, and \code{mpitb.set} function.
}
\author{
Ignacio Girela
}
