\name{autologistic}
\alias{autologistic}
\title{Fit a centered autologistic model using maximum pseudolikelihood estimation or MCMC for Bayesian inference.}
\usage{
  autologistic(formula, data, A, method = c("PL", "Bayes"),
    model = TRUE, x = FALSE, y = FALSE, verbose = FALSE,
    control = list())
}
\arguments{
  \item{formula}{an object of class \code{\link{formula}}:
  a symbolic description of the model to be fitted.}

  \item{data}{an optional data frame, list, or environment
  (or object coercible by \code{\link{as.data.frame}} to a
  data frame) containing the variables in the model. If not
  found in \code{data}, the variables are taken from
  \code{environment(formula)}, typically the environment
  from which \code{autologistic} is called.}

  \item{A}{the adjacency matrix for the underlying graph.}

  \item{method}{the method to use for inference.
  \dQuote{\code{PL}} (the default) enables maximum
  pseudolikelihood estimation, and \dQuote{\code{Bayes}}
  enables Bayesian inference.}

  \item{control}{a list of the following control
  parameters.  \describe{ \item{\code{confint}}{ (PL) the
  method for computing confidence intervals. The possible
  values are \dQuote{\code{sandwich}} (the default),
  \dQuote{\code{bootstrap}}, and \dQuote{\code{none}}.}
  \item{\code{sigma}}{ (Bayes) the common standard
  deviation for \eqn{\beta}'s prior distribution. Defaults
  to 1,000.} \item{\code{eta.max}}{ (Bayes) the right
  endpoint for \eqn{\eta}'s prior distribution. Defaults to
  2.} \item{\code{bootit}}{ (PL) the size of the bootstrap
  sample. This applies when \code{confint} is
  \dQuote{\code{sandwich}} or \dQuote{\code{bootstrap}},
  since samples from the fitted model are needed in both
  cases. Defaults to 1,000.} \item{\code{trainit}}{ (Bayes)
  the length of the training run. Defaults to 100,000.}
  \item{\code{minit}}{ (Bayes) the minimum number of
  posterior samples. Defaults to 100,000.}
  \item{\code{maxit}}{ (Bayes) the maximum number of
  posterior samples. Defaults to 1,000,000.}
  \item{\code{tol}}{ (Bayes) the tolerance. Defaults to
  0.01.} \item{\code{parallel}}{ (PL) a logical value
  indicating whether to parallelize the bootstrap. This
  defaults to \code{TRUE} if the \pkg{snow} package is
  installed.} \item{\code{type}}{ (PL) the cluster type,
  one of \dQuote{\code{SOCK}} (default),
  \dQuote{\code{PVM}}, \dQuote{\code{MPI}}, or
  \dQuote{\code{NWS}}.} \item{\code{nodes}}{ (PL) the
  number of slave nodes to create.}}}

  \item{model}{a logical value indicating whether the model
  frame should be included as a component of the returned
  value.}

  \item{x}{a logical value indicating whether the model
  matrix used in the fitting process should be returned as
  a component of the returned value.}

  \item{y}{a logical value indicating whether the response
  vector used in the fitting process should be returned as
  a component of the returned value.}

  \item{verbose}{a logical value indicating whether to
  print various messages to the screen, including progress
  updates when \code{method} is \dQuote{\code{Bayes}}.
  Defaults to \code{FALSE}.}
}
\value{
  \code{autologistic} returns an object of class
  \dQuote{\code{autologistic}}, which is a list containing
  the following components.  \item{coefficients}{the point
  estimate of \eqn{\theta}.} \item{fitted.values}{the
  fitted mean values, obtained by transforming the linear
  predictors by the inverse of the link function.}
  \item{linear.predictors}{the linear fit on link scale.}
  \item{residuals}{the response residuals.} \item{iter}{the
  size of the bootstrap/posterior sample.}
  \item{sample}{(where relevant) an \code{iter} by \eqn{p}
  matrix containing the bootstrap/posterior samples.}
  \item{mcse}{(where relevant) a \eqn{p}-vector of Monte
  Carlo standard errors.} \item{S}{(where relevant) the
  estimated sandwich matrix.} \item{accept}{(Bayes) the
  acceptance rate for the MCMC sampler.} \item{y}{if
  requested (the default), the \code{y} vector used.}
  \item{X}{if requested, the model matrix.} \item{model}{if
  requested (the default), the model frame.}
  \item{call}{the matched call.} \item{formula}{the formula
  supplied.} \item{method}{the method used for inference.}
  \item{convergence}{the integer code returned by
  \code{\link{optim}} subsequent to optimizing the
  pseudolikelihood.} \item{message}{a character string to
  go along with \code{convergence}.} \item{terms}{the
  \code{\link{terms}} object used.} \item{data}{the
  \code{data} argument.} \item{xlevels}{(where relevant) a
  record of the levels of the factors used in fitting.}
  \item{control}{a list containing the names and values of
  the control parameters.} \item{value}{the value of the
  negative log pseudolikelihood at its minimum.}
}
\description{
  Fit a centered autologistic model using maximum
  pseudolikelihood estimation or MCMC for Bayesian
  inference.
}
\details{
  This function fits the centered autologistic model of
  Caragea and Kaiser (2009) using maximum pseudolikelihood
  estimation or MCMC for Bayesian inference.  The joint
  distribution for the centered autologistic model is
  \deqn{\pi(Z\mid\theta)=c(\theta)^{-1}\exp\left(Z^\prime
  X\beta - \eta Z^\prime A\mu + \frac{\eta}{2}Z^\prime
  AZ\right),}{\pi(Z | \theta)=c(\theta)^{-1} exp(Z'X\beta -
  \eta Z'A\mu + 0.5 \eta Z'AZ),} where \eqn{\theta =
  (\beta^\prime, \eta)^\prime}{\theta = (\beta', \eta)'} is
  the parameter vector, \eqn{c(\theta)} is an intractable
  normalizing function, \eqn{Z} is the response vector,
  \eqn{X} is the design matrix, \eqn{\beta} is a
  \eqn{(p-1)}-vector of regression coefficients, \eqn{A} is
  the adjacency matrix for the underlying graph, \eqn{\mu}
  is the vector of independence expectations, and
  \eqn{\eta} is the spatial dependence parameter.  \cr \cr
  Maximum pseudolikelihood estimation sidesteps the
  intractability of \eqn{c(\theta)} by maximizing the
  product of the conditional likelihoods.  Confidence
  intervals are then obtained using a parametric bootstrap
  or a normal approximation, i.e., sandwich estimation. The
  bootstrap datasets are generated by perfect sampling
  (\code{\link{rautologistic}}).  The bootstrap samples can
  be generated in parallel using the \pkg{snow} package.
  \cr \cr Bayesian inference is obtained using the
  auxiliary variable algorithm of Moller et al. (2006).
  The auxiliary variables are generated by perfect
  sampling.  \cr \cr The prior distributions are (1)
  zero-mean normal with independent coordinates for
  \eqn{\beta}, and (2) uniform for \eqn{\eta}.  The common
  standard deviation for the normal prior can be supplied
  by the user as control parameter \code{sigma}. The
  default is 1,000. The uniform prior has support [0, 2] by
  default, but the right endpoint can be supplied (as
  control parameter \code{eta.max}) by the user.  \cr \cr
  The posterior covariance matrix of \eqn{\theta} is
  estimated using samples obtained during a training run.
  The default number of iterations for the training run is
  100,000, but this can be controlled by the user (via
  parameter \code{trainit}). The estimated covariance
  matrix is then used as the proposal variance for a
  Metropolis-Hastings random walk. The proposal
  distribution is normal. The posterior samples obtained
  during the second run are used for inference. The length
  of the run can be controlled by the user via parameters
  \code{minit}, \code{maxit}, and \code{tol}. The first
  determines the minimum number of iterations. If
  \code{minit} has been reached, the sampler will terminate
  when \code{maxit} is reached or all Monte Carlo standard
  errors are smaller than \code{tol}, whichever happens
  first. The default values for \code{minit}, \code{maxit},
  and \code{tol} are 100,000; 1,000,000; and 0.01,
  respectively.
}
\examples{
\dontrun{

# This is the continuation of the example given for function rautologistic.

# Find the MPLE, and do not compute confidence intervals.

fit = autologistic(Z ~ X - 1, A = A, control = list(confint = "none"))
summary(fit)

# Compute confidence intervals based on the normal approximation. Estimate the "filling" in the
# sandwich matrix using a parallel parametric bootstrap, where the computation is distributed
# across six cores on the host workstation.

fit = autologistic(Z ~ X - 1, A = A, verbose = TRUE,
                   control = list(confint = "sandwich", nodes = 6))
summary(fit)

# Compute confidence intervals based on a parallel parametric bootstrap. Use a bootstrap sample
# of size 500, and distribute the computation across six cores on the host workstation.

fit = autologistic(Z ~ X - 1, A = A, verbose = TRUE,
                  control = list(confint = "bootstrap", bootit = 500, nodes = 6))
summary(fit)

# Make some level plots of the residuals.

dev.new()
levelplot(residuals(fit) ~ x * y, aspect = "iso", col.regions = colfunc(n^2))
dev.new()
levelplot(residuals(fit, type = "pearson") ~ x * y, aspect = "iso", col.regions = colfunc(n^2))
dev.new()
levelplot(residuals(fit, type = "response") ~ x * y, aspect = "iso", col.regions = colfunc(n^2))

# Do MCMC for Bayesian inference. The length of the training run will be 10,000, and
# the length of the subsequent inferential run will be at least 10,000.

fit = autologistic(Z ~ X - 1, A = A, verbose = TRUE, method = "Bayes",
                  control = list(trainit = 10000, minit = 10000, sigma = 1000))
summary(fit)
}
}
\references{
  Caragea, P. and Kaiser, M. (2009) Autologistic models
  with interpretable parameters. \emph{Journal of
  Agricultural, Biological, and Environmental Statistics},
  \bold{14}(3), 281--300.

  Hughes, J., Haran, M. and Caragea, P. C. (2011)
  Autologistic models for binary data on a lattice.
  \emph{Environmetrics}, \bold{22}(7), 857--871.

  Moller, J., Pettitt, A., Berthelsen, K., and Reeves, R.
  (2006) An efficient Markov chain Monte Carlo method for
  distributions with intractable normalising constants.
  \emph{Biometrika}, \bold{93}(2), 451--458.
}
\seealso{
  \code{\link{rautologistic}},
  \code{\link{residuals.autologistic}},
  \code{\link{summary.autologistic}},
  \code{\link{vcov.autologistic}}
}

