% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BUGS_macros.R
\name{model_macro_builder}
\alias{model_macro_builder}
\title{EXPERIMENTAL: Turn a function into a model macro}
\usage{
model_macro_builder(fun, use3pieces = TRUE, unpackArgs = TRUE)
}
\arguments{
\item{fun}{A function written to construct new lines of model code (see below).}

\item{use3pieces}{logical indicating whether the arguments from the input
line be split into pieces for the LHS (left-hand side), RHS
(right-hand side, possibly further split depending on
\code{unpackArgs}), and \code{stoch} (\code{TRUE} if the line uses a
\code{~} and \code{FALSE} otherwise). (default = \code{TRUE})}

\item{unpackArgs}{logical indicating whether arguments be passed as a list
(\code{FALSE}) or as separate arguments (\code{TRUE}). (default = \code{TRUE})}
}
\value{
A list of class \code{model_macro} with one element called \code{process},
which contains the macro function suitable for use by \code{nimbleModel}.
}
\description{
A model macro expands one line of code in a nimbleModel into one or
more new lines.  This supports compact programming by defining
re-usable modules.  \code{model_macro_builder} takes as input a
function that constructs new lines of model code from the original
line of code.  It returns a function suitable for internal use by
\code{nimbleModel} that arranges arguments for input function.  Macros
are an experimental feature and are available only after setting
\code{nimbleOptions(enableModelMacros = TRUE)}.
}
\details{
The arguments \code{use3pieces} and \code{unpackArgs}
indicate how \code{fun} expects to have arguments arranged from an
input line of code (processed by \code{nimbleModel}).

Consider the defaults \code{use3pieces = TRUE} and \code{unpackArgs =
TRUE}, for a macro called \code{macro1}.  In this case, the line of
model code \code{x ~ macro1(arg1 = z[1:10], arg2 = "hello")} will be
passed to \code{fun} as \code{fun(stoch = TRUE, LHS = x, arg1 =
z[1:10], arg2 = "hello")}.

If \code{use3pieces = TRUE} but \code{unpackArgs = FALSE}, then the
RHS will be passed as is, without unpacking its arguments into
separate arguments to \code{fun}.  In this case, \code{x ~ macro1(arg1
= z[1:10], arg2 = "hello")} will be passed to \code{fun} as
\code{fun(stoch = TRUE, LHS = x, RHS = macro1(arg1 = z[1:10], arg2 =
"hello"))}.

If \code{use3pieces = FALSE} and \code{unpackArgs = FALSE}, the entire
line of code is passed as a single object.  In this case, \code{x ~
macro1(arg1 = z[1:10], arg2 = "hello")} will be passed to \code{fun}
as \code{fun(x ~ macro1(arg1 = z[1:10], arg2 = "hello"))}.  It is also
possible in this case to pass a macro without using a \code{~} or
\code{<-}.  For example, the line \code{macro1(arg1 = z[1:10], arg2 =
"hello")} will be passed to \code{fun} as \code{fun(macro1(arg1 =
z[1:10], arg2 = "hello"))}.

If \code{use3pieces = FALSE} and \code{unpackArgs = TRUE}, it
won't make sense to anticipate a declaration using \code{~} or \code{<-}.
Instead, arguments from an arbitrary call will be passed as separate arguments.
For example, the line \code{macro1(arg1 = z[1:10], arg2 = "hello")} will be
passed to \code{fun} as \code{fun(arg1 = z[1:10], arg2 = "hello")}.

In addition, the final two arguments of \code{fun} must be called \code{modelInfo}
and \code{.env} respectively. 

During macro processing, \code{nimbleModel} passes a named list to the \code{modelInfo} 
argument of \code{fun} containing, among other things, elements called
\code{constants} and \code{dimensions}. Macro developers can modify these
two elements (for example, to add a new constant needed for a macro) and
these changes will be reflected in the final model object. Note that currently
it is not possible for a macro to modify the data. Furthermore, if your macro add a new element to the
constants that \code{nimbleModel} then moves to the data, this new data will not be retained
in the final model object and thus will not be usable. 

\code{nimbleModel} passes the R environment from which \code{nimbleModel} was
called to the \code{.env} argument.

The \code{fun} function must return a named list with two elements:
\code{code}, the replacement code, and \code{modelInfo}, the \code{modelInfo} 
list described above. \code{modelInfo} must be in the output even if the macro
does not modify it.

It is extremely useful to be familiar with processing R code as an
object to write \code{fun} correctly.  Functions such as
\code{\link{substitute}} and \code{\link{as.name}}
(e.g. \code{as.name('~')}), \code{\link{quote}}, \code{\link{parse}}
and \code{\link{deparse}} are particularly handy.

Multiple lines of new code should be contained in \code{ {} }. Extra
curly braces are not a problem. See example 2.

Macro expansion is done recursively: One macro can return code that
invokes another macro.
}
\examples{
nimbleOptions(enableModelMacros = TRUE)
nimbleOptions(enableMacroComments = FALSE)
nimbleOptions(verbose = FALSE)

## Example 1: Say one is tired of writing "for" loops.
## This macro will generate a "for" loop with dnorm declarations
all_dnorm <- model_macro_builder(
    function(stoch, LHS, RHSvar, start, end, sd = 1, modelInfo, .env) {
        newCode <- substitute(
            for(i in START:END) {
                LHS[i] ~ dnorm(RHSvar[i], SD)
            },
            list(START = start,
                 END = end,
                 LHS = LHS,
                 RHSvar = RHSvar,
                 SD = sd))
        list(code = newCode)
    },
    use3pieces = TRUE,
    unpackArgs = TRUE 
)

model1 <- nimbleModel(
    nimbleCode(
    {
        ## Create a "for" loop of dnorm declarations by invoking the macro
        x ~ all_dnorm(mu, start = 1, end = 10)
    }
    ))

## show code from expansion of macro
model1$getCode()
## The result should be:
## {
##     for (i in 1:10) {
##         x[i] ~ dnorm(mu[i], 1)
##     }
## }

## Example 2: Say one is tired of writing priors.
## This macro will generate a set of priors in one statement
flat_normal_priors <- model_macro_builder(
    function(..., modelInfo, .env) {
        allVars <- list(...)
        priorDeclarations <- lapply(allVars,
                                    function(x)
                                        substitute(VAR ~ dnorm(0, sd = 1000),
                                                   list(VAR = x)))
        newCode <- quote({})
        newCode[2:(length(allVars)+1)] <- priorDeclarations
        list(code = newCode)
    },
    use3pieces = FALSE,
    unpackArgs = TRUE
)

model2 <- nimbleModel(
    nimbleCode(
    {
        flat_normal_priors(mu, beta, gamma)
    }
    ))

## show code from expansion of macro
model2$getCode()
## The result should be:
## {
##    mu ~ dnorm(0, sd = 1000)
##    beta ~ dnorm(0, sd = 1000)
##    gamma ~ dnorm(0, sd = 1000)
## }

## Example 3: Macro that modifies constants
new_constant <- model_macro_builder(
   function(stoch, LHS, RHS, modelInfo, .env) {
     # number of elements
     n <- as.numeric(length(modelInfo$constants[[deparse(LHS)]]))
     code <- substitute({
       for (i in 1:N){
         L[i] ~ dnorm(mu[i], 1)
       }
     }, list(L = LHS, N = n))

     # Add a new constant mu
     modelInfo$constants$mu <- rnorm(n, 0, 1)

     list(code = code, modelInfo = modelInfo)
   },
   use3pieces = TRUE,
   unpackArgs = TRUE
)

const <- list(y = rnorm(10))
code <- nimbleCode({
 y ~ new_constant()
})

mod <- nimbleModel(code = code, constants=const)
mod$getCode()
mod$getConstants() # new constant is here
}
