% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\docType{methods}
\name{handleFlags,ctd-method}
\alias{handleFlags,ctd-method}
\title{Handle Flags in CTD Objects}
\usage{
\S4method{handleFlags}{ctd}(object, flags = NULL, actions = NULL,
  debug = getOption("oceDebug"))
}
\arguments{
\item{object}{A \code{ctd} object, i.e. one inheriting from \code{\link{ctd-class}}.}

\item{flags}{A \code{\link{list}} specifying flag values upon which
actions will be taken. This can take two forms. In the first, the
list has named elements each containing a vector of integers. For example,
salinities flagged with values of 1 or 3 through 9 would be specified
by \code{flags=list(salinity=c(1,3:9))}. Several data items can be specified,
e.g. \code{flags=list(salinity=c(1,3:9), temperature=c(1,3:9))} indicates
that the actions are to take place for both salinity and temperature.
In the second form, \code{flags} is a list with unnamed vectors, and
this means to apply the actions to all the data entries; thus,
\code{flags=list(c(1,3:9))} means to apply not just to salinity and temperature,
but also to everything else that is in the \code{data} slot. If \code{flags}
is not provided, then \code{\link{defaultFlags}} is called, to try to
determine a conservative default.}

\item{actions}{An optional \code{\link{list}} that contains items with
names that match those in the \code{flags} argument.  If \code{actions}
is not supplied, the default will be to set all values identified by
\code{flags} to \code{NA}; this can also be specified by
specifying \code{actions=list("NA")}. It is also possible to specify
functions that calculate replacement values. These are provided
with \code{object} as the single argument, and must return a
replacement for the data item in question.
See \dQuote{Details} for the default that is used if \code{actions} is not supplied.}

\item{debug}{An optional integer specifying the degree of debugging, with
value 0 meaning to skip debugging and 1 or higher meaning to print some
information about the arguments and the data. It is usually a good idea to set
this to 1 for initial work with a dataset, to see which flags are being
handled for each data item. If not supplied, this defaults to the value of
\code{\link{getOption}("oceDebug")}.}
}
\description{
Data-quality flags are stored in the \code{metadata}
slot of \code{\link{oce-class}} objects in a
\code{\link{list}} named \code{flags}.
The present function (a generic that has specialized versions
for various data classes) provides a way to
manipulate the core data based on
the data-quality flags. For example, a common operation is to replace suspicious
or erroneous data with \code{NA}.

If \code{metadata$flags} in the object supplied as the first argument
is empty, then that object is returned, unaltered.
Otherwise, \code{handleFlags} analyses the data-quality flags within
the object, in relation to the \code{flags} argument, and interprets
the \code{action} argument to select an action to be applied to matched
data.
}
\examples{
library(oce)
data(section)
stn <- section[["station", 100]]
# 1. Default: anything not flagged as 2 is set to NA, to focus
# solely on 'good', in the World Hydrographic Program scheme.
STN1 <- handleFlags(stn, flags=list(c(1, 3:9)))
data.frame(old=stn[["salinity"]], new=STN1[["salinity"]], salinityFlag=stn[["salinityFlag"]])

# 2. Use bottle salinity, if it is good and ctd is bad
replace <- 2 == stn[["salinityBottleFlag"]] && 2 != stn[["salinityFlag"]]
S <- ifelse(replace, stn[["salinityBottle"]], stn[["salinity"]])
STN2 <- oceSetData(stn, "salinity", S)

# 3. Use smoothed TS relationship to nudge questionable data.
f <- function(x) {
  S <- x[["salinity"]]
  T <- x[["temperature"]]
  df <- 0.5 * length(S) # smooths a bit
  sp <- smooth.spline(T, S, df=df)
  0.5 * (S + predict(sp, T)$y)
}
par(mfrow=c(1,2))
STN3 <- handleFlags(stn, flags=list(salinity=c(1,3:9)), action=list(salinity=f))
plotProfile(stn, "salinity", mar=c(3, 3, 3, 1))
p <- stn[["pressure"]]
par(mar=c(3, 3, 3, 1))
plot(STN3[["salinity"]] - stn[["salinity"]], p, ylim=rev(range(p)))

}
\references{
1. \url{https://www.nodc.noaa.gov/woce/woce_v3/wocedata_1/whp/exchange/exchange_format_desc.htm}
}
\seealso{
Other functions relating to data-quality flags: \code{\link{defaultFlags}},
  \code{\link{handleFlags,adp-method}},
  \code{\link{handleFlags,argo-method}},
  \code{\link{handleFlags,section-method}},
  \code{\link{handleFlags}},
  \code{\link{initializeFlagScheme,ctd-method}},
  \code{\link{initializeFlagScheme,oce-method}},
  \code{\link{initializeFlagScheme,section-method}},
  \code{\link{initializeFlagScheme}},
  \code{\link{initializeFlags,adp-method}},
  \code{\link{initializeFlags,oce-method}},
  \code{\link{initializeFlags}},
  \code{\link{setFlags,adp-method}},
  \code{\link{setFlags,ctd-method}},
  \code{\link{setFlags,oce-method}}, \code{\link{setFlags}}

Other things related to \code{ctd} data: \code{\link{[[,ctd-method}},
  \code{\link{[[<-,ctd-method}}, \code{\link{as.ctd}},
  \code{\link{cnvName2oceName}}, \code{\link{ctd-class}},
  \code{\link{ctdDecimate}}, \code{\link{ctdFindProfiles}},
  \code{\link{ctdRaw}}, \code{\link{ctdTrim}},
  \code{\link{ctd}}, \code{\link{initialize,ctd-method}},
  \code{\link{initializeFlagScheme,ctd-method}},
  \code{\link{oceNames2whpNames}},
  \code{\link{oceUnits2whpUnits}},
  \code{\link{plot,ctd-method}}, \code{\link{plotProfile}},
  \code{\link{plotScan}}, \code{\link{plotTS}},
  \code{\link{read.ctd.itp}}, \code{\link{read.ctd.odf}},
  \code{\link{read.ctd.sbe}},
  \code{\link{read.ctd.woce.other}},
  \code{\link{read.ctd.woce}}, \code{\link{read.ctd}},
  \code{\link{setFlags,ctd-method}},
  \code{\link{subset,ctd-method}},
  \code{\link{summary,ctd-method}},
  \code{\link{woceNames2oceNames}},
  \code{\link{woceUnit2oceUnit}}, \code{\link{write.ctd}}
}
\concept{functions relating to data-quality flags}
\concept{things related to \code{ctd} data}
