% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/topo.R
\name{download.topo}
\alias{download.topo}
\title{Download and Cache a topo File}
\usage{
download.topo(
  west,
  east,
  south,
  north,
  resolution = 4,
  destdir = ".",
  destfile,
  format,
  server = "https://gis.ngdc.noaa.gov",
  debug = getOption("oceDebug")
)
}
\arguments{
\item{west, east}{numeric values for the limits of the data-selection box, in degrees.
These are converted to the -180 to 180 degree notation, if needed.
Then, \code{west} is rounded down to the nearest 1/100th degree, and \code{east}
is rounded up to the the nearest 1/100th degree. The results of these
operations are used in constructing the query for the NOAA data server.}

\item{south, north}{latitude limits, treated in a way that
corresponds to the longitude limits.}

\item{resolution}{numeric value of grid spacing, in geographical minutes.
The default value is 4 minutes, corresponding to 4 nautical miles (approx. 7.4km)
in the north-south direction, and less in the east-west direction.}

\item{destdir}{Optional string indicating the directory in which to store
downloaded files. If not supplied, \code{"."} is used, i.e. the data file
is stored in the present working directory.}

\item{destfile}{Optional string indicating the name of the file.
If not supplied, the file name is constructed from the other
parameters of the function call, so subsequent calls with the same
parameters will yield the same result, thus providing the key
to the caching scheme.}

\item{format}{Deprecated, and ignored, as of June 2020.}

\item{server}{character value specifying the base from which a
download URL will be constructed.  It is unlikely that any value
other than the default will work, unless it is a similarly-constructed
mirrored site.}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If one function calls another, it usually reduces the value of
\code{debug} first, so that a user can often obtain deeper debugging
by specifying higher \code{debug} values.}
}
\value{
String indicating the full pathname to the downloaded file.
}
\description{
Topographic data are downloaded from a data server that holds the ETOPO1
dataset (Amante, C. and B.W. Eakins, 2009), and saved as a netCDF file whose
name specifies the data request, if a file of that name is not already
present on the local file system.  The return value is the name of the data
file, and its typical use is as the filename for a call to \code{\link[=read.topo]{read.topo()}}.
Given the rules on file naming, subsequent calls to \code{download.topo()}
with identical parameters will simply return the name of the cached file,
assuming the user has not deleted it in the meantime.
}
\details{
The specified longitude and latitude limits are rounded to 2 digits
(corresponding to a footprint of approximately 1km), and these are used
in the server request. If the resultant request would generate under
1 row or column in the result, \link{download.topo} generates an
error message and stops.
}
\section{Historical note relating to NOAA server changes}{

In May of 2020, \code{\link[=download.topo]{download.topo()}} stopped working, evidently owing
to changes in the NOAA server API, which had been inferred by reverse
engineering a NOAA data-request website. Luckily, \code{\link[marmap:getNOAA.bathy]{marmap::getNOAA.bathy()}}
was found to be working at that time, and so \code{\link[=download.topo]{download.topo()}} was revised based on
that function.  The problem of keeping up with changing data-server APIs should
be easier in the future, since NOAA has made the API public.
}

\examples{
\dontrun{
library(oce)
topoFile <- download.topo(west=-66, east=-60, south=43, north=47,
                          resolution=1, destdir="~/data/topo")
topo <- read.topo(topoFile)
imagep(topo, zlim=c(-400, 400), col=oceColorsTwo, drawTriangles=TRUE)
if (requireNamespace("ocedata", quietly=TRUE)) {
    data(coastlineWorldFine, package="ocedata")
    lines(coastlineWorldFine[["longitude"]], coastlineWorldFine[["latitude"]])
}
}

}
\references{
\itemize{
\item Amante, C. and B.W. Eakins, 2009. ETOPO1 1 Arc-Minute Global Relief
Model: Procedures, Data Sources and Analysis. NOAA Technical Memorandum
NESDIS NGDC-24. National Geophysical Data Center, NOAA. doi:10.7289/V5C8276M
}
}
\seealso{
Other functions that download files: 
\code{\link{download.amsr}()},
\code{\link{download.coastline}()},
\code{\link{download.met}()}

Other things related to topo data: 
\code{\link{[[,topo-method}},
\code{\link{[[<-,topo-method}},
\code{\link{as.topo}()},
\code{\link{plot,topo-method}},
\code{\link{read.topo}()},
\code{\link{subset,topo-method}},
\code{\link{summary,topo-method}},
\code{\link{topo-class}},
\code{\link{topoInterpolate}()},
\code{\link{topoWorld}}
}
\author{
Dan Kelley
}
\concept{functions that download files}
\concept{things related to topo data}
