

################################################################################
#
# PREAMBLE
#


# This is preliminary example R code for using vegan, BiodiversityR and opm to
# draw PCA biplots describing Phenotype MicroArray data. An advantage of such
# plots is that the overall differences between the organisms can be plotted
# together with indications of the effect of certain subtrates on these 
# differences. Furthermore, the data are reduced to their independent 
# components before plotting. Depending on the biological question, this might
# be more interesting than drawing heat maps.
#
# (C) 2013 by Markus Goeker (markus [DOT] goeker [AT] dsmz [DOT] de)
#
# This file is distributed under the terms of the GPL. Hints by Pia Wuest, DSMZ,
# are gratefully acknowledged.


library(opm)
library(BiodiversityR)


################################################################################


#' PCA biplot of PM data with BiodiversityR
#' 
#' This function is intended to operate on numeric matrices as generated by
#' opm::extract. It uses methods from the vegan and BiodiversityR packages.
#'
#' @param x Matrix as exported by opm::extract.
#' @param group Logical scalar. Assign symbols by group? This causes secondary
#'   symbols to be plotted over the primary symbols (which should best be
#'   invisible in that case).
#' @param circle Logical scalar. Draw equlibrium circle?
#' @param text.col Used for drawing descriptor names.
#' @param points.col Used for drawing the primary object symbols.
#' @param arrow.col Used for drawing descriptor arrows.
#' @param legend.x Passed to graphics::legend in non-interactive use.
#' @param legend.y Likewise.
#' @param lwd Used for the secondary object symbols.
#' @param cex Used for drawing descriptor names.
#' @param scaling Passed to stats::biplot.
#' @param ... Optional arguments passed to stats::biplot.
#' 
#' @return Object of class "pca", returned invisibly.
#'
custom_pca <- function(x, group = TRUE, circle = scaling == 1,
    text.col = "darkgrey", points.col = if (group)
      "white"
    else
      "black", arrow.col = "red", legend.x = "topleft", legend.y = NULL,
    lwd = 2, cex = 0.4, scaling = 1, ...) {
  result <- vegan::rda(x) # conducts a PCA in this setting
  result.plot <- biplot(x = result, col = c(points.col, arrow.col),
    scaling = scaling, ...)
  if (group) {
    groups <- data.frame(.N = if (is.null(attr(x, "row.groups")))
        rownames(x)
      else
        attr(x, "row.groups"))
    if (can.click <- dev.capabilities()$locator)
      message("click on the preferred position of the legend")
    # Problem: in interactive use the returned color names have been incorrect!
    # This did not seem to work on an X11 device! PDF was OK but see below.
    cols.used <- BiodiversityR::ordisymbol(result.plot, groups, ".N",
      legend = can.click, lwd = lwd)
    if (!can.click)
      legend(legend.x, legend.y, legend = levels(groups$.N),
        text.col = cols.used)
  }
  text(result.plot, "species", col = text.col, cex = cex)
  if (circle)
    # Another problem: the circle does not appear in the PDF!
    # Only X11 worked so far.
    BiodiversityR::ordiequilibriumcircle(result, result.plot)
  invisible(result)
}


################################################################################


# The following code would create an PCA biplot for the vaas_4 example data.
#
if (FALSE) {
  x <- opm::extract(vaas_4, as.labels = list("Strain"),
    as.groups = list("Species"))
  x.pca <- custom_pca(x)
}


