% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameter_spaces.R
\name{ParameterSpaces}
\alias{ParameterSpaces}
\title{Switch Between Parameter Spaces}
\description{
This R6 object manages two related parameter spaces: the Optimization Space
(for optimization) and the Interpretation Space (for easier interpretation).

In the Optimization Space, parameters are stored as a \code{numeric}
\code{vector}, the standard format for numerical optimizers. Parameters in
this space are typically identified.

In the Interpretation Space, parameters are stored as a \code{list} and can
take different formats (e.g., \code{matrix}). Parameters here do not need to
be identified.

The user can define transformation functions (not necessarily bijective) to
switch between these two spaces via the \code{$o2i()} and \code{$i2o()}
methods.
}
\examples{
### Log-likelihood function of two-class Gaussian mixture model with
### parameter vector `theta` that consists of
### - `mu`, mean vector of length 2
### - `sd`, standard deviation vector of length 2, must be positive
### - `lambda`, class probability of length 1, must be between 0 and 1

normal_mixture_llk <- function(theta, data) {
  mu <- theta[1:2]
  sd <- exp(theta[3:4])
  lambda <- plogis(theta[5])
  c1 <- lambda * dnorm(data, mu[1], sd[1])
  c2 <- (1 - lambda) * dnorm(data, mu[2], sd[2])
  sum(log(c1 + c2))
}

### define parameter spaces
### - `mu` needs no transformation
### - `sd` needs to be real in optimization space and positive in
###    interpretation space
### - `lambda` needs to be real and of length `1` in optimization space, and
###    a probability vector of length `2` in interpretation space

normal_mixture_spaces <- ParameterSpaces$
  new(
    parameter_names = c("mu", "sd", "lambda"),
    parameter_lengths_in_o_space = c(2, 2, 1)
  )$
  o2i(
    "mu" = function(x) x,
    "sd" = function(x) exp(x),
    "lambda" = function(x) c(plogis(x), 1 - plogis(x))
  )$
  i2o(
    "mu" = function(x) x,
    "sd" = function(x) log(x),
    "lambda" = function(x) qlogis(x[1])
  )

### switch between parameter spaces

par <- list(                             # parameters in interpretation space
  "mu" = c(2, 4),
  "sd" = c(0.5, 1),
  "lambda" = c(0.4, 0.6)
)
(x <- normal_mixture_spaces$switch(par)) # switch to optimization space
normal_mixture_llk(
  theta = x, data = datasets::faithful$eruptions
)
normal_mixture_spaces$switch(x)          # switch back

}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-ParameterSpaces-new}{\code{ParameterSpaces$new()}}
\item \href{#method-ParameterSpaces-print}{\code{ParameterSpaces$print()}}
\item \href{#method-ParameterSpaces-switch}{\code{ParameterSpaces$switch()}}
\item \href{#method-ParameterSpaces-o2i}{\code{ParameterSpaces$o2i()}}
\item \href{#method-ParameterSpaces-i2o}{\code{ParameterSpaces$i2o()}}
\item \href{#method-ParameterSpaces-clone}{\code{ParameterSpaces$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ParameterSpaces-new"></a>}}
\if{latex}{\out{\hypertarget{method-ParameterSpaces-new}{}}}
\subsection{Method \code{new()}}{
Initializes a new \code{ParameterSpaces} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ParameterSpaces$new(parameter_names, parameter_lengths_in_o_space)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{parameter_names}}{[\code{character()}]\cr
Unique names for the parameters.}

\item{\code{parameter_lengths_in_o_space}}{[\code{integer()}]\cr
The length of each parameter in the optimization space.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{ParameterSpaces} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ParameterSpaces-print"></a>}}
\if{latex}{\out{\hypertarget{method-ParameterSpaces-print}{}}}
\subsection{Method \code{print()}}{
Print an overview of the parameter spaces.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ParameterSpaces$print(show_transformer = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{show_transformer}}{[\code{logical(1)}]\cr
Show transformer functions in the output?}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ParameterSpaces-switch"></a>}}
\if{latex}{\out{\hypertarget{method-ParameterSpaces-switch}{}}}
\subsection{Method \code{switch()}}{
Switch between Optimization Space and Interpretation Space.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ParameterSpaces$switch(x, to = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{[\code{numeric()} | \code{list()}]\cr
The parameters, either as a \verb{numeric vector} (will be switched to
Interpretation Space), or as a \code{list()} (will be switched to Optimization
Space).}

\item{\code{to}}{[\code{character(1)} | \code{NULL}]\cr
Explicitly switch to a specific space, either
\itemize{
\item \code{"o"}: Optimization Space
\item \code{"i"}: Interpretation Space
}

If \code{NULL}, the function will switch to the other space.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ParameterSpaces-o2i"></a>}}
\if{latex}{\out{\hypertarget{method-ParameterSpaces-o2i}{}}}
\subsection{Method \code{o2i()}}{
Define transformation functions when switching from Optimization Space to
Interpretation Space.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ParameterSpaces$o2i(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{[\code{function}]\cr
One or more transformation functions, named according to the parameters.

Transformers from Optimization Space to Interpretation Space (o2i)
\strong{must receive} a \code{numeric}. The default is the identity.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ParameterSpaces-i2o"></a>}}
\if{latex}{\out{\hypertarget{method-ParameterSpaces-i2o}{}}}
\subsection{Method \code{i2o()}}{
Define transformation functions when switching from Interpretation Space
to Optimization Space.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ParameterSpaces$i2o(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{[\code{function}]\cr
One or more transformers functions, named according to the parameters.

Transformers from Interpretation Space to Optimization Space (i2o)
\strong{must return} a \code{numeric}. The default is \code{as.vector()}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ParameterSpaces-clone"></a>}}
\if{latex}{\out{\hypertarget{method-ParameterSpaces-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ParameterSpaces$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
