% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outliertree.R
\name{outlier.tree}
\alias{outlier.tree}
\title{Outlier Tree}
\usage{
outlier.tree(df, cols_ord = NULL, cols_ignore = NULL,
  save_outliers = FALSE, outliers_print = 10, max_depth = 4,
  min_gain = 0.1, z_norm = 2.67, z_outlier = 8,
  pct_outliers = 0.01, min_size_numeric = 25, min_size_categ = 75,
  categ_as_bin = TRUE, ord_as_bin = TRUE,
  cat_bruteforce_subset = FALSE, follow_all = FALSE, nthreads = -1)
}
\arguments{
\item{df}{Data Frame with normal data that might contain some outliers.}

\item{cols_ord}{Character vector indicating which categorical columns are ordinal.
Ordinal columns must be passed as factors.}

\item{cols_ignore}{Vector containing columns which will not be split, but will be evaluated for usage
in splitting other columns. Can pass either a logical (boolean) vector with the same number of columns
as `df`, or a character vector of column names (must match with those of `df`).
Pass `NULL` to use all columns.}

\item{save_outliers}{Whether to store outliers detected in `df` in the object that is returned.
These outliers can then be extracted from the returned object through function
`extract.training.outliers`.}

\item{outliers_print}{Maximum number of flagged outliers in the training data to print after fitting
the model. Pass zero or `NULL` to avoid printing any. Outliers can be printed from resulting data frame
afterwards through the `predict` method, or through the `print` method (on the extracted outliers, not on
the model object) if passing `save_outliers` = `TRUE`.}

\item{max_depth}{Maximum depth of the trees to grow. Can also pass zero, in which case it will only look
for outliers with no conditions (i.e. takes each column as a 1-d distribution and looks for outliers in
there independently of the values in other columns).}

\item{min_gain}{Minimum gain that a split has to produce in order to consider it (both in terms of looking
for outliers in each branch, and in considering whether to continue branching from them). Note that default
value for GritBot is 1e-6.}

\item{z_norm}{Maximum Z-value (from standard normal distribution) that can be considered as a normal
observation. Note that simply having values above this will not automatically flag observations as outliers,
nor does it assume that columns follow normal distributions. Also used for categorical and ordinal columns
for building approximate confidence intervals of proportions.}

\item{z_outlier}{Minimum Z-value that can be considered as an outlier. There must be a large gap in the
Z-value of the next observation in sorted order to consider it as outlier, given by (z_outlier - z_norm).
Ignored for categorical and ordinal columns.}

\item{pct_outliers}{Approximate max percentage of outliers to expect in a given branch.}

\item{min_size_numeric}{Minimum size that branches need to have when splitting a numeric column.}

\item{min_size_categ}{Minimum size that branches need to have when splitting a categorical or ordinal column.}

\item{categ_as_bin}{Whether to make categorical-by-categorical binary splits by binarizing each category
in the column and then attempting splits by grouping categories into subsets. Alternative is to create
one branch per category of the column being split from. Ignored when there is only one or fewer categorical
columns. Can only pass one of `categ_as_bin` and `cat_bruteforce_subset`.}

\item{ord_as_bin}{Same as `categ_as_bin`, but for ordinal columns, and cumulative (i.e. it splits by "<=",
not "="). Ignored when there are no ordinal columns or no categorical columns.}

\item{cat_bruteforce_subset}{Whether to make categorical-by-categorical binary splits by trying all the
possible combinations of columns in each subset (that is, it evaluates 2^n potential splits every time).
Note that trying this when there are many categories in a column will result in exponential computation
time that might never finish. Alternative is to create one branch per category of the column being split
from. Ignored when there is only one or fewer categorical columns. Can only pass one of `categ_as_bin`
and `cat_bruteforce_subset`.}

\item{follow_all}{Whether to continue branching from each split that meets the size and gain criteria.
This will produce exponentially many more branches, and if depth is large, might take forever to finish.
Will also produce a lot more spurious outiers. Not recommended.}

\item{nthreads}{Number of parallel threads to use. When fitting the model, it will only use up to one
thread per column, while for prediction it will use up to one thread per row. The more threads that are
used, the more memory will be required and allocated, so using more threads will not always lead to better
speed. Passing zero or negative numbers will default to the maximum number of available CPU cores (but not
if the object attribute is overwritten). Can be changed after the object is already initialized.}
}
\value{
An object with the fitted model that can be used to detect more outliers in new data, and from
which outliers in the training data can be extracted (when passing `save_outliers` = `TRUE`).
}
\description{
Fit Outlier Tree model to normal data with perhaps some outliers.
}
\details{
Explainable outlier detection through decision-tree grouping. Tries to detect outliers by
generating decision trees that attempt to "predict" the values of each column based on each other column,
testing in each branch of every tried split (if it meets some minimum criteria) whether there are
observations that seem too distant from the others in a 1-D distribution for the column that the split
tries to "predict" (unlike other methods, this will not generate a score for each observation).

Splits are based on gain, while outlierness is based on confidence intervals. Similar in spirit to the GritBot
software developed by RuleQuest research.

Supports columns of types numeric, categorical, and ordinal (for this last one, will consider their order when
splitting other columns from them, but not when splitting to "predict" them), and can handle missing values
in any of them. Can also pass dates/timestamps that will get converted to numeric but shown as dates/timestamps
in the output. Offers option to set columns to be used only to split other columns but not to look at outliers
in them.

Infinite values will be taken into consideration when the column is used to split another column
(that is, +inf will go into the branch that is greater than something, -inf into the other branch),
but when a column is the target of the split, they will be taken as missing - that is, it will not report
infinite values as outliers.
}
\examples{
library(outliertree)

### example dataset with interesting outliers
data(hypothyroid)

### fit the model and get a print of outliers
model <- outlier.tree(hypothyroid, outliers_print=10, save_outliers=TRUE)

### extract outlier info as R list
outliers <- extract.training.outliers(model)
summary(outliers)

### information for row 745
outliers[[745]]

### use custom row names
df.w.names <- hypothyroid
row.names(df.w.names) <- paste0("rownum", 1:nrow(hypothyroid))
outliers.w.names <- predict(model, df.w.names, return_outliers=TRUE)
outliers.w.names[["rownum745"]]
}
\references{
GritBot software: \url{https://www.rulequest.com/gritbot-info.html}
}
\seealso{
\link{predict.outliertree} \link{extract.training.outliers} \link{hypothyroid}
}
