% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_models.R
\name{pcsslm}
\alias{pcsslm}
\title{Approximate a linear model using PCSS}
\usage{
pcsslm(formula, pcss = list(), ...)
}
\arguments{
\item{formula}{an object of class formula whose dependent variable is a 
combination of variables and logical | operators. 
All model terms must have appropriate PCSS in \code{pcss}.}

\item{pcss}{a list of precomputed summary statistics. In all cases, this
should include \code{n}: the sample size, \code{means}: a named vector of
predictor and response means, and \code{covs}: a named covariance matrix
including all predictors and responses. See Details for more information.}

\item{...}{additional arguments. See Details for more information.}
}
\value{
an object of class \code{"pcsslm"}.

  An object of class \code{"pcsslm"} is a list containing at least the 
  following components:
    \item{call}{the matched call}
    \item{terms}{the \code{terms} object used}
    \item{coefficients}{a \eqn{p x 4} matrix with columns for the 
      estimated coefficient, its standard error, t-statistic and
      corresponding (two-sided) p-value.}
    \item{sigma}{the square root of the estimated variance of the random
      error.}
    \item{df}{degrees of freedom, a 3-vector \eqn{p, n-p, p*}, the
      first being the number of non-aliased coefficients, the last being
      the total number of coefficients.}
    \item{fstatistic}{a 3-vector with the value of the F-statistic with its
      numerator and denominator degrees of freedom.}
    \item{r.squared}{\eqn{R^2}, the 'fraction of variance explained by the 
      model'.}
    \item{adj.r.squared}{the above \eqn{R^2} statistic \emph{'adjusted'},
      penalizing for higher \eqn{p}.}
    \item{cov.unscaled}{a \eqn{p x p} matrix of (unscaled) covariances of the
      \eqn{coef[j], j=1,...p}.}
    \item{Sum Sq}{a 3-vector with the model's Sum of Squares Regression 
      (SSR), Sum of Squares Error (SSE), and Sum of Squares Total (SST).}
}
\description{
\code{pcsslm} approximates a linear model of a combination of variables using
  precomputed summary statistics.
}
\details{
\code{pcsslm} parses the input \code{formula}'s dependent variable for 
  functions such as sums (\code{+}), products (\code{*}), or logical 
  operators (\code{|} and \code{&}).
  It then identifies models the combination of variables using one of
  \code{\link{model_combo}}, \code{\link{model_product}}, 
  \code{\link{model_or}}, \code{\link{model_and}}, or 
  \code{\link{model_prcomp}}.

  Different precomputed summary statistics are needed inside \code{pcss}
  depending on the function that combines the dependent variable.
  \itemize{
    \item For linear combinations (and principal component analysis), only
          \code{n}, \code{means}, and \code{covs} are required
    \item For products and logical combinations, the additional items
          \code{predictors} and \code{responses} are required.
          These are named lists of objects of class \code{predictor}
          generated by \code{\link{new_predictor}}, with a \code{predictor}
          object for each independent variable in \code{predictors} and
          each dependent variable in \code{responses}.
          However, if only modeling the product or logical combination of 
          only two variables, \code{responses} can be \code{NULL} without
          consequence.
  }
  
  If modeling a principal component score of a set of variables, include 
  the argument \code{comp} where \code{comp} is an integer indicating which 
  principal component score to analyze. Optional logical arguments 
  \code{center} and \code{standardize} determine if responses should be
  centered and standardized before principal components are calculated. 
  
  If modeling a linear combination, include the argument \code{phi}, a named
  vector of linear weights for each variable in the dependent variable in 
  formula.
  
  If modeling a product, include the argument \code{response}, a character
  equal to either \code{"continuous"} or \code{"binary"}. If \code{"binary"},
  specialized approximations are performed to estimate means and variances.
}
\examples{
## Principal Component Analysis
ex_data <- pcsstools_example[c("g1", "x1", "y1", "y2", "y3")]
pcss <- list(
  means = colMeans(ex_data),
  covs = cov(ex_data),
  n = nrow(ex_data)
)

pcsslm(y1 + y2 + y3 ~ g1 + x1, pcss = pcss, comp = 1)

## Linear combination of variables
ex_data <- pcsstools_example[c("g1", "g2", "y1", "y2")]
pcss <- list(
  means = colMeans(ex_data),
  covs = cov(ex_data),
  n = nrow(ex_data)
)

pcsslm(y1 + y2 ~ g1 + g2, pcss = pcss, phi = c(1, -1))
summary(lm(y1 - y2 ~ g1 + g2, data = ex_data))

## Product of variables
ex_data <- pcsstools_example[c("g1", "x1", "y4", "y5", "y6")]

pcss <- list(
  means = colMeans(ex_data),
  covs = cov(ex_data),
  n = nrow(ex_data),
  predictors = list(
    g1 = new_predictor_snp(maf = mean(ex_data$g1) / 2),
    x1 = new_predictor_normal(mean = mean(ex_data$x1), sd = sd(ex_data$x1))
  ),
  responses = lapply(
    colMeans(ex_data)[3:length(colMeans(ex_data))], 
    new_predictor_binary
  )
)

pcsslm(y4 * y5 * y6 ~ g1 + x1, pcss = pcss, response = "binary")
summary(lm(y4 * y5 * y6 ~ g1 + x1, data = ex_data))

## Disjunct (OR statement) of variables
ex_data <- pcsstools_example[c("g1", "x1", "y4", "y5")]

pcss <- list(
  means = colMeans(ex_data),
  covs = cov(ex_data),
  n = nrow(ex_data),
  predictors = list(
    g1 = new_predictor_snp(maf = mean(ex_data$g1) / 2),
    x1 = new_predictor_normal(mean = mean(ex_data$x1), sd = sd(ex_data$x1))
  )
)
pcsslm(y4 | y5 ~ g1 + x1, pcss = pcss) 
summary(lm(y4 | y5 ~ g1 + x1, data = ex_data))

}
\references{
{

  \insertRef{wolf_using_2021}{pcsstools}
  
  \insertRef{wolf_computationally_2020}{pcsstools}
  
  \insertRef{gasdaska_leveraging_2019}{pcsstools}

}
}
\seealso{
\code{\link{model_combo}}, \code{\link{model_product}}, 
  \code{\link{model_or}}, \code{\link{model_and}}, and 
  \code{\link{model_prcomp}}.
}
