% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/invwavtrans.R
\name{InvWavTransf2D}
\alias{InvWavTransf2D}
\title{Inverse average-interpolation 2D wavelet transform}
\usage{
InvWavTransf2D(D, M0, order = c(3, 3), jmax, metric = "Riemannian",
  progress = T, ...)
}
\arguments{
\item{D}{a list of arrays containing the 2D pyramid of wavelet coefficients, where each array contains the
(\eqn{d,d})-dimensional wavelet coefficients from the finest wavelet scale \code{j = jmax} up to the coarsest
wavelet scale \code{j = 0}. This is the same format as the \code{$D} component given as output by
 \code{\link{WavTransf2D}}.}

\item{M0}{a numeric array containing the midpoint(s) at the coarsest scale \code{j = 0} in the 2D midpoint pyramid.
This is the same format as the \code{$M0} component given as output by \code{\link{WavTransf2D}}.}

\item{order}{a 2-dimensional numeric vector of odd integers larger or equal to 1 corresponding to the marginal
orders of the intrinsic 2D AI refinement scheme, defaults to \code{order = c(5, 5)}. Note that if \code{max(order) > 9},
the computational cost significantly increases as the wavelet transform no longer uses a fast wavelet refinement scheme based
on pre-determined weights.}

\item{jmax}{the maximum scale (resolution) up to which the 2D surface of HPD midpoints (i.e. scaling coefficients) are
reconstructed. If \code{jmax} is not specified it is set equal to the resolution in the finest wavelet scale
\code{jmax = length(D)}.}

\item{metric}{the metric that the space of HPD matrices is equipped with. The default choice is \code{"Riemannian"},
but this can also be one of: \code{"logEuclidean"}, \code{"Cholesky"}, \code{"rootEuclidean"} or
\code{"Euclidean"}. The inverse intrinsic 2D AI wavelet transform fundamentally relies on the chosen metric.}

\item{progress}{should a console progress bar be displayed? Defaults to \code{progress = T}.}

\item{...}{additional arguments for internal use.}
}
\value{
Returns a (\eqn{d, d, n_1, n_2})-dimensional array corresponding to a rectangular surface of size \eqn{n_1} by
\eqn{n_2} of (\eqn{d,d})-dimensional HPD matrices.
}
\description{
\code{InvWavTransf2D} computes the inverse intrinsic average-interpolation (AI) wavelet
transform of an array of coarsest-scale HPD midpoints combined with a 2D pyramid of Hermitian
wavelet coefficients. This is the inverse operation of the function \code{\link{WavTransf2D}}.
}
\details{
The input list of arrays \code{D} and array \code{M0} correspond to a 2D pyramid of wavelet coefficients and
the coarsest-scale HPD midpoints respectively, both are structured in the same way as in the output of
\code{WavTransf2D}. As in the forward AI wavelet transform, if both marginal refinement orders are
smaller or equal to 9, the function computes the inverse wavelet transform using a fast wavelet refinement scheme based
on weighted geometric averages with pre-determined weights. If one of the marginal refinement order is an odd integer
larger than 9, the wavelet refinement scheme is based on intrinsic polynomial surface prediction using Neville's algorithm on the
Riemannian manifold (\code{\link{pdNeville}}). By default \code{InvWavTransf2D} computes the inverse intrinsic 2D AI wavelet transform
equipping the space of HPD matrices with (i) the Riemannian metric. Instead, the space of HPD matrices can also be
equipped with one of the following metrics; (ii) log-Euclidean metric, the Euclidean inner product between matrix logarithms,
(iii) Cholesky metric, the Euclidean inner product between Cholesky decompositions, (iv) Euclidean metric and
(v) root-Euclidean metric. The default choice (Riemannian) has several appealing properties not shared by the
other metrics, see (Chau and von Sachs, 2017a) for more details.
}
\examples{
P <- rExamples2D(c(2^4, 2^4), 2, example = "tvar")
P.wt <- WavTransf2D(P$f) ## forward transform
P.f <- InvWavTransf2D(P.wt$D, P.wt$M0) ## backward transform
all.equal(P.f, P$f)

}
\references{
Chau, J. and von Sachs, R. (2017) \emph{Positive definite multivariate spectral
estimation: a geometric wavelet approach}. Available at \url{http://arxiv.org/abs/1701.03314}.
}
\seealso{
\code{\link{WavTransf2D}}, \code{\link{pdSpecEst2D}}, \code{\link{pdNeville}}
}
