#' plot.phenology plots the phenology.
#' @title Plot the phenology from a result.
#' @author Marc Girondot
#' @return Return A list of lists with formated outputs\cr
#' For each site:\cr
#' $site=name of the site\cr
#' $estimate1=Estimation of counts not taking into account the observations\cr
#' $sd1=The SD of estimation1\cr
#' $estimate2=Estimation of counts taking into account the observations\cr
#' $sd2=The SD of estimation2
#' @param x A result file generated by fit_phenology
#' @param ... Parameters used by plot
#' @param data A dataset generated by add_format
#' @param parametersfixed Set of fixed parameters to be replaced. Must be a named vector
#' @param parameters Set of parameters to be replaced. Must be a named vector
#' @param se Set of SE values. Must be a named vector
#' @param series Number of series to be analyzed or 'all'
#' @param moon If TRUE, the moon phase is ploted. Default is FALSE
#' @param replicate.CI Number of replicates for estimation of confidence interval
#' @param progressbar If FALSE, do not show the progress bar
#' @param help If TRUE, an help is displayed
#' @param growlnotify If False, does not send growl notification
#' @description The function plot.phenology plots the phenology graph from a result object.\cr
#' If cofactors have been added, the plot does not show their effects.
#' @examples
#' \dontrun{
#' library(phenology)
#' # Read a file with data
#' Gratiot<-read.delim("http://max2.ese.u-psud.fr/epc/conservation/BI/Complete.txt", header=FALSE)
#' data(Gratiot)
#' # Generate a formatted list nammed data_Gratiot 
#' data_Gratiot<-add_phenology(Gratiot, name="Complete", 
#' 		reference=as.Date("2001-01-01"), format="%d/%m/%Y")
#' # Generate initial points for the optimisation
#' parg<-par_init(data_Gratiot, parametersfixed=NULL)
#' # Run the optimisation
#' result_Gratiot<-fit_phenology(data=data_Gratiot, 
#' 		parametersfit=parg, parametersfixed=NULL, trace=1)
#' data(result_Gratiot)
#' # Plot the phenology and get some stats
#' output<-plot(result_Gratiot)
#' # Plot only part of the nesting season
#' ptoutput<-plot(result_Gratiot, xlim=c(as.Date("2001-03-01"),as.Date("2001-08-31")))
#' # Use month names in English
#' Sys.setlocale(category = "LC_TIME", locale="en_GB.UTF-8")
#' output<-plot(result_Gratiot)
#' # set back the month name in local R language
#' Sys.setlocale(category = "LC_TIME", locale="")
#' }
#' @method plot phenology
#' @export

#plot.phenology <- function(x, ...) {

plot.phenology <- 
	function(x, ..., data=NULL, parameters=NULL, parametersfixed=NULL, se=NULL, 
	series="all", moon=FALSE, replicate.CI=1000, 
	progressbar=TRUE, help=FALSE, growlnotify=TRUE) {

# data=NULL; parameters=NULL; parametersfixed=NULL; se=NULL; series="all"; moon=FALSE; replicate.CI=1000; progressbar=TRUE; help=FALSE; growlnotify=TRUE
    result <- x
    


out <- list()

if (is.null(data)) data <- result$data

if (series[1]=="all") series <- c(1:length(data))

series <- as.numeric(series)

# kseries <- 1
for(kseries in seq_along(series)) {

  reference <- data[[series[kseries]]]$Date[1]-data[[series[kseries]]]$ordinal[1]+1
  res_se <- se
  parres <- parameters
  parametersfixed <- parametersfixed
  
if (!is.null(result)) {
  res_se <- unlist(modifyList(as.list(result$se), as.list(se)))
	parres <- unlist(modifyList(as.list(result$par), as.list(parres)))
	parametersfixed <- unlist(modifyList(as.list(result$parametersfixed), as.list(parametersfixed)))
}

res_se[is.na(res_se)] <- 0


# length(result$par) donne le nombre de paramètres pour lesquels on a result
# length(parametersfixed) donne le nombre de paramètres fixés
# je passe les parametersfixed en revue pour savoir si le sd# est fourni pour ce paramètre

if (!is.null(parametersfixed)) {
if (!all(is.na(parametersfixed))) {

for(i in seq_along(parametersfixed)) {
	nm <- names(parametersfixed[i])
# est-ce que j ai un sd pour ce paramètre et le paramètre
# si oui je stocke les deux
# sinon, je mets le sd à 0

	if (substr(nm,1,3)!="sd#")	{
		parres <- c(parres, parametersfixed[i])
		if (all(names(res_se) != nm)) {
		if (is.na(parametersfixed[paste0("sd#", nm)])) {
# je n'ai pas de sd pour ce paramètre
		  ik <- 0
		  names(ik) <- nm
			res_se <- c(res_se, ik)
		} else {
# j'ai un sd pour ce paramètre	
		  ik <- parametersfixed[paste0("sd#", nm)]
		  names(ik) <- nm
			res_se <- c(res_se, ik)
		}
		}
	}
}
}
}

## crée un tableau dans lequel on met ce qu'on va utiliser pour le graph
val=matrix(c(1:366, rep(NA, 366), rep(NA, 366), rep(NA, 366), rep(NA, 366), rep(NA, 366), rep(NA, 366)), ncol=7)

## On le nomme, c'est plus simple
colnames(val)=c("days", "Obs", "Theor", "Theor-2SE", "Theor+2SE", "Theor-2SD", "Theor+2SD")

if (is.null(data)) {nmser=""} else {
	nmser <- names(data[series[kseries]])
# 2012-06-03 Je rentre les valeurs avec un nombre
	xxx <- data[series[kseries]][[1]][,"ordinal"]
	val[xxx,"Obs"] <- data[series[kseries]][[1]][,"nombre"]
	
	# 2012-06-03 Je rentre les valeurs avec une incertitude
	for (xjkk in seq_along(data[series[kseries]][[1]][,"ordinal2"])) {
		if (!is.na(data[series[kseries]][[1]][xjkk,"ordinal2"])) {
			xxx <- seq(from=data[series[kseries]][[1]][xjkk,"ordinal"], to=data[series[kseries]][[1]][xjkk,"ordinal2"])
			val[xxx,"Obs"] <- data[series[kseries]][[1]][xjkk,"nombre"]/length(xxx)
		}
	}

	
}

if (!is.null(parres)) {

## crée un tableau avec des réplicats des modèles
par2=matrix(rep(NA, length(parres)*replicate.CI), ncol=length(parres))

## On nomme les colonnes comme les noms des paramètres
colnames(par2)=names(parres)


# on générère replicate.CI jeux de paramètres à partir des paramètres obtenus en vraisemblance
# ici il faut que je prenne aussi ceux transmis en parmetersfixed
for(i in names(parres)) par2[,i]=rnorm(replicate.CI, mean=as.numeric(parres[i]), sd=res_se[i])

## je génère une matrice de replicate.CI saisons de pontes
## note que je ne gère pas les années bissextiles
ponte2=matrix(rep(NA, 366*replicate.CI), ncol=366)

# affiche le nom de la série
# message("\n", nmser, "\n")

if (progressbar) pb<-txtProgressBar(min=1, max=replicate.CI, style=3)



## je remplis les replicate.CI saisons de ponte
for(j in 1:replicate.CI) {

if (progressbar) setTxtProgressBar(pb, j)

# j'ai tous les paramètres dans xpar
# maintenant tous les paramètre fixés appraissent dans resfit
#	xparec <- phenology:::.format_par(par2[j,], nmser)
	xparec <- getFromNamespace(".format_par", ns="phenology")(par2[j,], nmser)
	
#	xparec <<- xparec
	
#	ponte2[j,1:365] <- phenology:::.daily_count(1:365, xparec, print=FALSE)
	ponte2[j,1:366] <- getFromNamespace(".daily_count", ns="phenology")(1:366, xparec, print=FALSE)
	
# je viens de générer les pontes du réplicat j
}

## je calcule les écart-types des nb de pontes générées chaque jour
sd2<-apply(ponte2, 2, sd)

mnponte<-mean(apply(ponte2, 1, sum))
sdponte<-sd(apply(ponte2, 1, sum))
out1<-c(estimate1=mnponte, sd1=sdponte)

# si c'est une donnée observée avec une série de dates, je mets à 0 la série
# dans ponte2[nbsimul 1:replicate.CI, jour 1:365] j'ai la donnée théorique
for(i in 1:dim(data[[series[kseries]]])[1]) {
		if (!is.na(data[[series[kseries]]]$ordinal2[i])) {
			#for(j in (1+data[[series[kseries]]]$ordinal[i]):data[[series[kseries]]]$ordinal2[i]) {
					ponte2[1:replicate.CI, (data[[series[kseries]]]$ordinal[i]):data[[series[kseries]]]$ordinal2[i]] <- 0
			#}
		}
}

nobs <- data[[series[kseries]]]$nombre
dobs <- data[[series[kseries]]]$ordinal
l <- length(dobs)
for(i in 1:l) {
	ponte2[1:replicate.CI, dobs[i]] <- nobs[i]
}

mnponte<-mean(apply(ponte2, 1, sum))
sdponte<-sd(apply(ponte2, 1, sum))
out1<-c(out1, estimate2=mnponte, sd2=sdponte)

# 20 mai 2012
out1 <- c(out1, CI_Min=max(mnponte-2*sdponte, sum(data[[series[kseries]]]$nombre)), CI_Max=mnponte+2*sdponte)

## je remplis le tableau val avec les nb théoriques

#xparec <- phenology:::.format_par(parres, nmser)
xparec <- getFromNamespace(".format_par", ns="phenology")(parres, nmser)

# val[1:365, "Theor"]=phenology:::.daily_count(1:365, xparec, print=FALSE)
val[1:366, "Theor"] <- getFromNamespace(".daily_count", ns="phenology")(1:366, xparec, print=FALSE)


## je remplis le tableau val avec les nb théoriques +/- 2 SD
for(i in 1:366) {val[i, "Theor-2SE"]=max(0, val[i, "Theor"]-2*sd2[i])}
val[1:366, "Theor+2SE"]=val[1:366, "Theor"]+2*sd2[1:366]


## je calcule la distribution théorique des points minimaux
for(i in 1:366) {val[i, "Theor-2SD"]=max(c(subset(0:trunc(3*val[i, "Theor"]), pnbinom(0:trunc(3*val[i, "Theor"]), size=abs(as.numeric(xparec["Theta"])), mu=val[i, "Theor"])<=0.05), 0))}

## je calcule la distribution théoriques des points maximaux
for(i in 1:366) {val[i, "Theor+2SD"]=min(c(subset(0:trunc(3*val[i, "Theor"]), pnbinom(0:trunc(3*val[i, "Theor"]), size=abs(as.numeric(xparec["Theta"])), mu=val[i, "Theor"])>=0.95), trunc(3*val[i, "Theor"])+1))}


}

vmaxx <- c(reference, reference+365)


if ((!is.null(data)) && (!is.null(parres))) {
	vmaxy <- c(0, max(val[, "Theor+2SD"], data[[series[kseries]]]$nombre[(is.na(data[[series[kseries]]]$ordinal2)) & (!is.na(data[[series[kseries]]]$nombre))]))
} else {
	if (!is.null(data)) {
		vmaxy <- c(0, max(data[[series[kseries]]]$nombre[(is.na(data[[series[kseries]]]$ordinal2)) & (!is.na(data[[series[kseries]]]$nombre))]))
	} else {
		vmaxy <- c(0, max(val[, "Theor+2SD"]))
	}
}

if (vmaxy[2]==0) vmaxy[2] <- 0.1


if (as.POSIXlt(reference+365)$year != as.POSIXlt(reference)$year) {
  # annee non-bisextile
  x <- seq(from=reference, to=reference+364, by="1 day")
} else {
  x <- seq(from=reference, to=reference+365, by="1 day")
  }


# xlab<-ifelse(is.null(pnp$xlab), "Months", pnp$xlab)
# ylab<-ifelse(is.null(pnp$ylab), "Number", pnp$ylab)
# pch<-ifelse(is.null(pnp$pch), 16, pnp$pch)
# cex<-ifelse(is.null(pnp$cex), 0.5, pnp$cex)

if (moon) {
	moony<-vmaxy[2]*1.06
	mp<-moon.info(x, phase=TRUE)
	mpT1<-ifelse((mp!="FM") | (is.na(mp)), FALSE, TRUE)
	mpT2<-ifelse((mp!="NM") | (is.na(mp)), FALSE, TRUE)
#	mpT3<-ifelse((mp!="FQ") | (is.na(mp)), FALSE, TRUE)
#	mpT4<-ifelse((mp!="LQ") | (is.na(mp)), FALSE, TRUE)
}


## je fais les graphiques
## Pour les dates seules
par(new=FALSE)


# pnp <- modifyList(list(xlab="Months", ylab="Counts", main=names(data[series[kseries]]), pch=16, cex=0.5, xlim=vmaxx, ylim=vmaxy, type="n", bty="n"), list())

pnp <- modifyList(list(xlab="Months", ylab="Counts", main=names(data[series[kseries]]), 
	pch=16, cex=0.5, xlim=vmaxx, ylim=vmaxy, type="n", bty="n"), list(...))
	

do.call(plot, modifyList(pnp, list(x=x, y=rep(0, length(x)))))


if (moon) {
	points(x[mpT1], rep(moony, length(x[mpT1])), cex=1, bg="black", col="black", pch=21, xpd=TRUE)
	points(x[mpT2], rep(moony, length(x[mpT2])), cex=1, bg="white", col="black", pch=21, xpd=TRUE)
#	points(x[mpT3], rep(moony, length(x[mpT3])), cex=3, bg="black", col="black", pch=21)	
#	points(x[mpT3]+8, rep(moony, length(x[mpT3])), cex=3, bg="white", col="white", pch=21)
#	points(x[mpT4], rep(moony, length(x[mpT4])), cex=3, bg="black", col="black", pch=21)	
#	points(x[mpT4]-8, rep(moony, length(x[mpT4])), cex=3, bg="white", col="white", pch=21)
}

par(new=TRUE)


if (!is.null(data)) {


pnp2 <- modifyList(pnp, list(xlab="", ylab="", main="", axes=FALSE, col="black", type="p"))

	
do.call(plot, modifyList(pnp2, list(x=data[[series[kseries]]]$Date[is.na(data[[series[kseries]]]$Date2)], y=data[[series[kseries]]]$nombre[is.na(data[[series[kseries]]]$Date2)])))



## Pour les dates avec incertitudes
par(new=TRUE)
for(i in 1:dim(data[[series[kseries]]])[1]) {
	if (!is.na(data[[series[kseries]]]$ordinal2[i])) {
		x0<-data[[series[kseries]]]$Date[i]
		x1<-data[[series[kseries]]]$Date2[i]
		lgt01<-as.numeric(data[[series[kseries]]]$Date2[i]-data[[series[kseries]]]$Date[i]+1)
		y0<-data[[series[kseries]]]$nombre[i]/lgt01
		y1<-y0
		segments(x0, y0, x1=x1, y1=y1, col="green", lwd=2)
	}
}



par(new=TRUE)
}

if (!is.null(parres)) {

pnp3 <- modifyList(pnp, list(xlab="", ylab="", main="", axes=FALSE, col="black", type="l"))


do.call(plot, modifyList(pnp3, list(x=reference+val[, "days"], y=val[, "Theor"])))
par(new=TRUE)
pnp3 <- modifyList(pnp, list(xlab="", ylab="", main="", axes=FALSE, col="black", type="l", lty=2))
do.call(plot, modifyList(pnp3, list(x=reference+val[, "days"], y=val[, "Theor-2SE"])))
par(new=TRUE)
do.call(plot, modifyList(pnp3, list(x=reference+val[, "days"], y=val[, "Theor+2SE"])))
par(new=TRUE)
pnp3 <- modifyList(pnp, list(xlab="", ylab="", main="", axes=FALSE, type="l", lty=2, col="red"))
do.call(plot, modifyList(pnp3, list(x=reference+val[, "days"], y=val[, "Theor-2SD"])))
par(new=TRUE)
do.call(plot, modifyList(pnp3, list(x=reference+val[, "days"], y=val[, "Theor+2SD"])))

}


if (!is.null(parres)) {

out2 <- list(list(estimates=out1, values=val))
names(out2) <- nmser

out <- c(out, out2)

}
# mtext(names(data[series[kseries]]), side=3, line=1)


}



	if (growlnotify) growlnotify('Plot is done!')
	
	class(out) <- "phenologyout"
	
	return(invisible(out))

}
