\name{picasso.logit}
\alias{picasso.logit}

\title{
Sparse Logistic Regression
}

\description{
The function "picasso.logit" implements sparse logistic regression.
}

\usage{
picasso.logit(X, Y, lambda = NULL, nlambda = NULL, lambda.min.ratio = NULL,
              method="l1", alg = "cyclic", gamma = 3, gr = NULL, gr.n = NULL, 
              gr.size = NULL, prec = 1e-4, max.ite = 1e4, verbose = TRUE)
}

\arguments{
  \item{Y}{The \eqn{n} dimensional response vector.}
  \item{X}{The \eqn{n} by \eqn{d} design matrix.}
  \item{lambda}{A sequence of decresing positive value to control the regularization. Typical usage is to leave the input \code{lambda = NULL} and have the program compute its own \code{lambda} sequence based on \code{nlambda} and \code{lambda.min.ratio}. Users can also specify a sequence to override this. Default value is from \eqn{lambda.max} to \code{lambda.min.ratio*lambda.max}. The default value of \eqn{lambda.max} is the minimum regularization parameter which yields an all-zero estiamtes.}
  \item{nlambda}{The number of values used in \code{lambda}. Default value is 5.}
  \item{lambda.min.ratio}{The smallest value for \code{lambda}, as a fraction of the uppperbound (\code{MAX}) of the regularization parameter. The program can automatically generate \code{lambda} as a sequence of length = \code{nlambda} starting from \code{MAX} to \code{lambda.min.ratio*MAX} in log scale. The default value is \code{0.25}.}
  \item{method}{Options for regularization. L1 is applied if \code{method = "l1"}, MCP is applied if \code{method = "scad"}, SCAD logit is applied if \code{method = "scad"} and group logit is applied if \code{method = "glasso"}. The default value is \code{"l1"}.}
  \item{alg}{Options for active set identification. Cyclic search is applied if \code{method = "cyclic"}, greedy search is applied if \code{method = "greedy"}, proximal gradient based search is applied if \code{method = "prox"} and stochastic search is applied if \code{method = "stoc"}. The default value is \code{"cyclic"}.}
  \item{gamma}{Parameter for MCP and SCAD. The default value is 3.}
  \item{gr}{A list which defines the grouping of the variables. The default setting is grouping every two variables if \code{method = "glogit"}.}
  \item{gr.n}{The number of groups. The default value is \code{method = "d/2"}.}
  \item{gr.size}{A vector of the number of variables in each group. The default value is a vector of 2 of size d/2.}
  \item{prec}{Stopping criterion. The default value is 1e-4.}
  \item{max.ite}{The iteration limit. The default value is 1e4.}
  \item{verbose}{Tracing information is disabled if \code{verbose = FALSE}. The default value is \code{TRUE}.}
  }
\details{ 
\deqn{
    \min_{\beta} {\frac{1}{n}}\sum_{i=1}^n (\log(1+e^{x_i^T \beta}) - y_i x_i^T \beta) + \lambda R(\beta),
    }\cr
    where \eqn{R(\beta)} can be \eqn{\ell_1} norm, MCP, SCAD or group regularization.
}
\value{
An object with S3 class \code{"picasso.logit"} is returned: 
  \item{beta}{
  A matrix of regression estimates whose columns correspond to regularization parameters.
  }
  \item{intercept}{The value of intercepts corresponding to regularization parameters. }
  \item{Y}{The value of \code{Y} used in the program. }
  \item{X}{The value of \code{X} used in the program. }
  \item{beta}{A \eqn{d \times nlambda} matrix of estimate of parameters.}
  \item{intercept}{A \eqn{1 \times nlambda} vector of estimate of intercept.}
  \item{lambda}{
  The sequence of regularization parameters \code{lambda} used in the program.
  }
  \item{nlambda}{
  The number of values used in \code{lambda}.
  }
  \item{method}{
  The \code{method} from the input. 
  }
  \item{alg}{
  The \code{alg} from the input. 
  }
  \item{df}{
  The degree of freecom (number of nonzero coefficients) along the solution path. 
  }
  \item{ite}{
  A list of vectors where the i-th entries of ite[[1]] and ite[[2]] correspond to the outer iteration and inner iteration of i-th regularization parameter respectively. 
  }
  \item{verbose}{The \code{verbose} from the input.}
}

\author{
Xingguo Li, Tuo Zhao and Han Liu \cr
Maintainer: Xingguo Li <xingguo.leo@gmail.com>
}

\references{
1. J. Friedman, T. Hastie and H. Hofling and R. Tibshirani. Pathwise coordinate optimization. \emph{The Annals of Applied Statistics}, 2007.\cr
2. C.H. Zhang. Nearly unbiased variable selection under minimax concave penalty. \emph{Annals of Statistics}, 2010.\cr
3. J. Fan and R. Li. Variable selection via nonconcave penalized likelihood and its oracle
properties. \emph{Journal of the American Statistical Association}, 2001.\cr
4. M. Yuan and Y. Lin. Model selection and estimation in the gaussian graphical model. \emph{Biometrika
}, 2007
}

\seealso{
\code{\link{picasso-package}}.
}

\examples{
## Generate the design matrix and regression coefficient vector
n = 100
d = 400
X = matrix(rnorm(n*d), n, d)
beta = c(3,2,0,1.5,rep(0,d-4))

## Generate response and fit sparse logistic models
p = exp(X\%*\%beta)/(1+exp(X\%*\%beta))
Y = rbinom(n,rep(1,n),p)
out.l1.cyclic = picasso.logit(X = X, Y = Y, nlambda = 10)

out.l1.greedy = picasso.logit(X = X, Y = Y, nlambda = 10, alg = "greedy")

out.mcp.cyclic = picasso.logit(X = X, Y = Y, nlambda = 10, method = "mcp")

out.glasso.prox = picasso.logit(X = X, Y = Y, nlambda = 10, method = "glasso", alg = "prox")

## Visualize the solution path
plot(out.l1.cyclic)
plot(out.l1.greedy)
plot(out.mcp.cyclic)
plot(out.glasso.prox)

## Estimate of Bernoulli parameters
p.l1 = out.l1.cyclic$p
}