% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/counterfactual_plot_continuous.R
\name{counterfactual.plot.continuous}
\alias{counterfactual.plot.continuous}
\title{Create a plot of the distribution of exposure under counterfactual 
  scenario for continuous exposure}
\usage{
counterfactual.plot.continuous(X, cft, weights = rep(1/nrow(as.matrix(X)),
  nrow(as.matrix(X))), adjust = 1, n = 512, ktype = c("gaussian",
  "epanechnikov", "rectangular", "triangular", "biweight", "cosine",
  "optcosine"), bw = c("SJ", "nrd0", "nrd", "ucv", "bcv"),
  title = "Exposure distribution under current and counterfactual scenarios",
  dnames = c("Current distribution", "Counterfactual distribution"),
  legendtitle = "Scenario", xlab = "Exposure", ylab = "Density",
  fill_limits = c(-Inf, Inf), fill = TRUE, colors = c("deepskyblue",
  "tomato3"), check_exposure = TRUE)
}
\arguments{
\item{X}{Univariate \code{vector} continuous exposure levels.}

\item{cft}{Counterfactual function of the exposure \code{cft(X)}.
  
\strong{**Optional**}}

\item{weights}{Normalized survey \code{weights} for the sample \code{X}.}

\item{adjust}{Adjust bandwith parameter from density from
\code{\link[stats]{density}}.}

\item{n}{Number of equally spaced points at which the density is to be
estimated (see \code{\link[stats]{density}}).}

\item{ktype}{\code{kernel} type:  \code{"gaussian"}, 
\code{"epanechnikov"}, \code{"rectangular"}, \code{"triangular"}, 
\code{"biweight"}, \code{"cosine"}, \code{"optcosine"} (for \code{kernel} 
method). Additional information on kernels in \code{\link[stats]{density}}.}

\item{bw}{Smoothing bandwith parameter from density from
\code{\link[stats]{density}}. Default \code{"SJ"}.}

\item{title}{String with plot title.}

\item{dnames}{String vector indicating the names of the distributions for 
the legend.}

\item{legendtitle}{String title for the legend of plot.}

\item{xlab}{String label for the X-axis of the plot (corresponding 
to "a").}

\item{ylab}{String label for the Y-axis of the plot (corresponding 
to "b").}

\item{fill_limits}{Vector. Limits of subset of the exposure \code{X} such
that only \code{fill_limits[1] < X < fill_limits[2]} are filled with color.}

\item{fill}{Boolean that indicates whether there is interior colouring. Default \code{TRUE}.}

\item{colors}{String vector with colors for plots.}

\item{check_exposure}{Check that exposure \code{X} is positive and numeric.}
}
\value{
cft_plot   \code{\link[ggplot2]{ggplot}} object plotting the shift
  from actual to counterfactual distribution.
}
\description{
Function that creates a plot of the distribution of exposure
  \code{X} under counterfactual scenario when exposure is continuous.
}
\note{
This function reproduces the classic counterfactual plot from Figure 
  25.1 of Vander Hoorn as well as additional plots.
}
\examples{

#Example 1: Normal distribution and linear counterfactual
#--------------------------------------------------------
set.seed(2783569)
X   <- data.frame(rnorm(1000, 150, 15))
cft <- function(X){0.35*X + 75}  
counterfactual.plot.continuous(X, cft, xlab = "Usual SBP (mmHg)", 
ylab = "Relative risk of ischaemic heart disease",
dnames = c("Current distribution", "Theoretical Minimum Risk Distribution"),
title = paste0("Effect of a non-linear hazard function and choice", 
               "\\nof baseline on total population risk", 
               "\\n(Fig 25 from Vander Hoorn et al)"))
  
#Example 2: Counterfactual of BMI reduction only for those with excess-weight (BMI > 25)
#--------------------------------------------------------
set.seed(2783569)
X <- data.frame(Exposure = rlnorm(1000, 3, 0.2))
cft <- function(X){

     #Find individuals with excess weight
     excess_weight <- which(X[,"Exposure"] > 25)
     
     #Set those with excess weight to BMI of 25
     X[excess_weight, "Exposure"] <- 22.5
     
     return(X)
}     

counterfactual.plot.continuous(X, cft, ktype = "epanechnikov")   

#Change bandwidth method to reduce noice
counterfactual.plot.continuous(X, cft, ktype = "epanechnikov", bw = "nrd0")   

#Focus on what happens to the exposure > 23 
counterfactual.plot.continuous(X, cft, ktype = "epanechnikov", bw = "nrd0",
fill_limits = c(23, Inf)) 

#Delete fill
counterfactual.plot.continuous(X, cft, ktype = "epanechnikov", bw = "nrd0", fill = FALSE)   
  
}
\references{
Vander Hoorn, S., Ezzati, M., Rodgers, A., Lopez, A. D., & 
  Murray, C. J. (2004). \emph{Estimating attributable burden of disease from 
  exposure and hazard data. Comparative quantification of health risks: 
  global and regional burden of disease attributable to selected major risk 
  factors}. Geneva: World Health Organization, 2129-40.
}
\seealso{
\code{\link{counterfactual.plot.discrete}} for plotting discrete counterfactuals, 
  \code{\link{pif}} for Potential Impact Fraction estimation, 
  \code{\link{pif.heatmap}} for sensitivity analysis of the counterfactual, 
  \code{\link{pif.plot}} for a plot of potential impact fraction as a 
  function of the relative risk's parameter \code{theta}.
}
\author{
Rodrigo Zepeda-Tello \email{rzepeda17@gmail.com}

Dalia Camacho-García-Formentí \email{daliaf172@gmail.com}
}
\keyword{internal}
