library(pipenostics)

test_that("*m325nhl* errs in normative heat loss", {
  expect_equal(
    m325nhl(1980, "underground", d = 73, temperature = 65, beta = c(FALSE, TRUE)),
    c(65.500, 75.325),
    tolerance = 1e-3
  )
  expect_equal(
    m325nhl(2000, "channel", d = 73, temperature = 65, beta = c(FALSE, TRUE)),
    c(17.533, 21.040),
    tolerance = 1e-3
  )

  with(m325nhldata, {
    expect_equal(
      m325nhl(epoch, laying, exp5k, insulation, as.double(diameter), temperature),
      loss
    )
  })
})

test_that("*m325beta* errs in calculation", {
  data(m325nhldata)
  expect_equal(
    unique(m325beta(m325nhldata[["laying"]], as.double(m325nhldata[["diameter"]]))),
    c(1.15, 1.2)
  )
})

test_that("*m325traceline* errs in tracing regime parameters", {
  m325traceline_foward_report <- m325traceline(130, .588399, 250, seq(0, 30, 10), forward = TRUE)
  expect_equal(
    names(m325traceline_foward_report),
    c("temperature", "pressure", "flow_rate", "loss", "flux", "Q")
  )
  expect_equal(
   m325traceline_foward_report[["temperature"]],
   c(129.1799, 128.4269, 127.9628, 127.3367),
   tolerance = 1e-4
  )
  expect_equal(
    m325traceline_foward_report[["pressure"]],
    c(0.5878607, 0.5874226, 0.5872143, 0.5870330),
    tolerance = 1e-7
  )
  expect_equal(
    m325traceline_foward_report[["flow_rate"]],
    c(250, 240, 220, 190)
  )
  expect_equal(
    m325traceline_foward_report[["loss"]],
    c(348, 347.138912477, 346.348251588, 345.860965187),
    tolerance = 1e-7
  )
  expect_equal(
    m325traceline_foward_report[["flux"]],
    c(181.959958158, 181.509718360, 181.096302779, 180.841513660),
    tolerance = 1e-7
  )
  expect_equal(
    m325traceline_foward_report[["Q"]],
    c(5011200, 4415607.97, 2493707.41, 2905232.11),
    tolerance = 1e-1
  )

  m325traceline_backward_report <- m325traceline(127.3367, .5870330, 190, seq(0, 30, 10), forward = FALSE)
  expect_equal(
    names(m325traceline_backward_report),
    c("temperature", "pressure", "flow_rate", "loss", "flux", "Q")
  )
  expect_equal(
    m325traceline_backward_report[["temperature"]],
    c(129.9953, 129.1769, 128.4254, 127.9619),
    tolerance = 1e-4
  )
  expect_equal(
    m325traceline_backward_report[["pressure"]],
    c(0.5883998, 0.5878611, 0.5874228, 0.5872144),
    tolerance = 1e-7
  )
  expect_equal(
    m325traceline_backward_report[["flow_rate"]],
    c(250, 250, 240, 220)
  )
  expect_equal(
    m325traceline_backward_report[["loss"]],
    c(347.135781269, 346.346651110, 345.859948477, 345.203535),
    tolerance = 1e-7
  )
  expect_equal(
    m325traceline_backward_report[["flux"]],
    c(181.508081135, 181.095465931, 180.840982050, 180.497760875),
    tolerance = 1e-7
  )
  expect_equal(
    m325traceline_backward_report[["Q"]],
    c(4998755.25, 4405529.40, 2490192.63, 2899710.69),
    tolerance = 1e-1
  )
})


test_that("*m325tracefw* errs in calculation without execution parallelization", {
  m325tracefw_report <- m325tracefw(verbose = FALSE)
  expect_equal(
    m325tracefw_report[["node"]],
    c("1", "2")
  )
  expect_equal(
    m325tracefw_report[["tracing"]],
    c("sensor", "1")
  )
  expect_equal(
    m325tracefw_report[["backward"]],
    c(FALSE, FALSE)
  )
  expect_equal(
    m325tracefw_report[["aggregation"]],
    c("identity", "identity")
  )
  expect_equal(
    m325tracefw_report[["temperature"]],
    c(70, 69.71603),
    tolerance = 1e-5
  )
  expect_equal(
    m325tracefw_report[["pressure"]],
    c(.5883990, 0.5813153)
  )
  expect_equal(
    m325tracefw_report[["flow_rate"]],
    c(20, 20)
  )
  expect_equal(
    m325tracefw_report[2, "loss"],
    78.4
  )
  expect_equal(
    m325tracefw_report[2, "flux"],
    279.06962283,
    tolerance = 1e-5
  )
  expect_equal(
    m325tracefw_report[2, "Q"],
    136314.3936,
    tolerance = 1e-1
  )
  expect_equal(
    m325tracefw_report[["job"]],
    c(0, 1)
  )
})

m325tracefw_report <- m325tracefw(verbose = FALSE, use_cluster = !nzchar(Sys.getenv("_R_CHECK_LIMIT_CORES_", "")))
test_that("*m325tracefw* errs in calculation utilizing parallel execution (if possible)", {
  expect_equal(
    m325tracefw_report[["node"]],
    c("1", "2")
  )
  expect_equal(
    m325tracefw_report[["tracing"]],
    c("sensor", "1")
  )
  expect_equal(
    m325tracefw_report[["backward"]],
    c(FALSE, FALSE)
  )
  expect_equal(
    m325tracefw_report[["aggregation"]],
    c("identity", "identity")
  )
  expect_equal(
    m325tracefw_report[["temperature"]],
    c(70, 69.71603),
    tolerance = 1e-5
  )
  expect_equal(
    m325tracefw_report[["pressure"]],
    c(.5883990, 0.5813153)
  )
  expect_equal(
    m325tracefw_report[["flow_rate"]],
    c(20, 20)
  )
  expect_equal(
    m325tracefw_report[2, "loss"],
    78.4
  )
  expect_equal(
    m325tracefw_report[2, "flux"],
    279.06962283,
    tolerance = 1e-5
  )
  expect_equal(
    m325tracefw_report[2, "Q"],
    136314.3936,
    tolerance = 1e-1
  )
  expect_equal(
    m325tracefw_report[["job"]],
    c(0, 1)
  )
})

test_that("*m325tracefw* does not write csv-file", {
  file_name <- tempfile()
  m325tracefw(csv = TRUE, file = file_name)
  expect_equal(
    file.exists(file_name),
    TRUE
  )
  unlink(file_name)
})

# Test m325tracebw

DHN <- pipenostics::m325testbench
DHN[["d"]] <- 1e3*DHN[["d"]]  # convert [m] to [mm]
m325tracebw_report <- do.call("m325tracebw", DHN)

test_that("*m325tracebw* errs in loss or in balance calculation", {
  with(subset(DHN, acceptor == 2), {
    expect_equal(
      temperature +
        dropt(
          temperature = temperature,
          pressure = pressure,
          flow_rate = flow_rate,
          loss_power = pipenostics::m325nhl(
            year = year,
            laying = laying,
            exp5k = exp5k,
            insulation = insulation,
            d = d,
            temperature = temperature,
            len = len,
            duration = 1,  # [hour]
            beta = beta,
            extra = 2
          )  # [kcal/hour]
        )
      ,
      subset(m325tracebw_report, node == 4 & tracing == 2 & aggregation == "identity",
             "temperature")[[1]]
    )

    expect_equal(
        pressure +
        dropp(
          temperature, pressure, flow_rate,
          d*1e-3, len, roughness, inlet, outlet, "romeo"
        ),
        subset(m325tracebw_report, node == 4 & tracing == 2 & aggregation == "identity",
               "pressure")[[1]]
    )
  })
})

test_that("*m325tracebw* errs in balance calculation", {
  expect_equal(
    subset(
       m325tracebw_report, node == 6 & aggregation == "median", "flow_rate"
    )[1, ],
    unname(colSums(subset(DHN, acceptor %in% c(3, 7, 9), "flow_rate")))
  )
})  

test_that("*m325tracebw* does not write csv-file", {
  file_name <- tempfile()
  m325tracebw(csv = TRUE, file = file_name)
  expect_equal(
    file.exists(file_name),
    TRUE
  )
  unlink(file_name)
})

test_that("*m325tracebw* does not produce ensample results", {
  aggregation_method <- c("span", "median", "mean", "identity")
  ensample_1 <- structure(
    list(
      node = c(
        "1", "2", "3", "7", "9", "10", "15",
        "17", "19", "21", "24", "25", "26", "1", "10", "15", "17", "19",
        "2", "21", "24", "25", "26", "3", "7", "9", "1", "10", "15",
        "17", "19", "2", "21", "24", "25", "26", "3", "7", "9", "1",
        "10", "15", "17", "19", "2", "21", "24", "25", "26", "3", "7",
        "9", "4", "4", "5", "6", "6", "11", "14", "16", "20", "20", "22",
        "22", "23", "14", "22", "23", "4", "5", "14", "22", "23", "4",
        "5", "14", "22", "23", "4", "5", "8", "6", "13", "18", "20",
        "20", "6", "20", "6", "20", "6", "8", "18", "18", "8", "18",
        "8", "18", "8", "11", "16", "11", "16", "11", "16", "11", "16",
        "13", "13", "13", "13", "13", "12", "12", "12", "12"
      ),
      tracing = c(
        rep.int("sensor", 52),
        "1", "2", "3", "7", "9", "10", "15", "17",
        "19", "21", "24", "25", "26", "15", "24|25", "26", "1|2", "3",
        "15", "24|25", "26", "1|2", "3", "15", "24|25", "26", "1|2",
        "3", "4", "5", "14", "22", "23", "19|21|23", "7|9|5", "19|21|23",
        "7|9|5", "19|21|23", "7|9|5", "6", "20", "22|20", "4|6", "22|20",
        "4|6", "22|20", "4|6", "8", "18", "10|8", "17|18", "10|8", "17|18",
        "10|8", "17|18", "11", "16", "14|11|16", "14|11|16", "14|11|16",
        "13", "13", "13", "13"
      ),
      backward = rep.int(TRUE, 116),
      aggregation = c(
        rep.int(tail(aggregation_method, 1), 13),
        rep(aggregation_method, each = 13),
        rep(aggregation_method, each = 5),
        rep.int(rep(aggregation_method, each = 2), 3),
        aggregation_method,
        head(aggregation_method, -1)
      ),
      loss = c(
        96.236, 96.288, 70.584, 78.4, 28.1152, 24.9182,
        96.6, 24.9596, 28.1658, 77.824, 96.184, 96.236, 70.54, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 96.236, 24.9182, 96.6, 24.9596,
        28.1658, 96.288, 77.824, 96.184, 96.236, 70.54, 70.584, 78.4,
        28.1152, 96.236, 24.9182, 96.6, 24.9596, 28.1658, 96.288, 77.824,
        96.184, 96.236, 70.54, 70.584, 78.4, 28.1152, 96.236, 96.288,
        70.584, 78.4, 28.1152, 24.9182, 96.6, 24.9596, 28.1658, 77.824,
        96.184, 96.236, 70.54, 0, 0.0520000000000067, 0, 0.0519999999999925,
        0, 96.6, 96.21, 70.54, 96.262, 70.584, 96.6, 96.21, 70.54, 96.262,
        70.584, 116.044943125762, 70.7340165868372, 96.7331745004449,
        115.945514486784, 70.6899252508703, 49.6582, 50.2848, 70.6899252508703,
        70.7340165868372, 58.8932417502901, 59.0830721956124, 96.2114863150603,
        96.1226072815915, 19.8229072051922, 19.8334568107018, 106.034060884188,
        106.128214720411, 106.034060884188, 106.128214720411, 116.015881478503,
        115.922822950485, 91.0976814785032, 90.9632229504852, 70.4670407392516,
        70.4412114752426, 70.4670407392516, 70.4412114752426, 116.67905024913,
        116.666828288858, 19.9458757486849, 116.666828288858, 110.026351012811,
        153.133682893198, 0, 153.133682893198, 153.133682893198
      ),
      flux = c(
        230.589178320317, 230.713774492983, 314.816883884811, 279.069622830087,
        125.398669015616, 174.048642246555, 231.461351528977, 174.337812964705,
        125.624353792967, 277.01931539705, 230.464582147651, 230.589178320317,
        314.620636252332, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 230.589178320317,
        174.048642246555, 231.461351528977, 174.337812964705, 125.624353792967,
        230.713774492983, 277.01931539705, 230.464582147651, 230.589178320317,
        314.620636252332, 314.816883884811, 279.069622830087, 125.398669015616,
        230.589178320317, 174.048642246555, 231.461351528977, 174.337812964705,
        125.624353792967, 230.713774492983, 277.01931539705, 230.464582147651,
        230.589178320317, 314.620636252332, 314.816883884811, 279.069622830087,
        125.398669015616, 230.589178320317, 230.713774492983, 314.816883884811,
        279.069622830087, 125.398669015616, 174.048642246555, 231.461351528977,
        174.337812964705, 125.624353792967, 277.01931539705, 230.464582147651,
        230.589178320317, 314.620636252332, 0, 0.124596172665719, 0,
        0.12459617266569, 0, 231.461351528977, 230.526880233984, 314.620636252332,
        230.65147640665, 314.816883884811, 231.461351528977, 230.526880233984,
        314.620636252332, 230.65147640665, 314.816883884811, 208.539746692486,
        315.485983884799, 231.78044831844, 208.36106744436, 315.289328878064,
        189.664975085096, 190.087314869183, 277.01931539705, 279.069622830087,
        239.310999356027, 239.984758576834, 230.530441564136, 230.317480267973,
        21.9564128236139, 21.9906948716501, 219.339273856167, 219.535094128311,
        219.339273856167, 219.535094128311, 208.487521163354, 208.320289387994,
        34.4388789167982, 33.9824764232883, 191.268081704955, 191.32905117635,
        191.268081704955, 191.32905117635, 209.679275355442, 209.657311757283,
        22.1231365611576, 209.679275355442, 217.039011810389, 184.655477184946,
        0, 184.655477184946, 184.655477184946
      ),
      Q = c(
        92028.56208, 92078.28864, 130903.392384,
        136314.3936, 20296.2504192, 13092.8196624, 184405.536, 13114.5725472,
        35131.9909824, 135312.900096, 58760.72928, 58792.49712, 130821.79104,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92028.56208, 13092.8196624,
        184405.536, 13114.5725472, 35131.9909824, 92078.28864, 135312.900096,
        58760.72928, 58792.49712, 130821.79104, 130903.392384, 136314.3936,
        20296.2504192, 92028.56208, 13092.8196624, 184405.536, 13114.5725472,
        35131.9909824, 92078.28864, 135312.900096, 58760.72928, 58792.49712,
        130821.79104, 130903.392384, 136314.3936, 20296.2504192, 92028.56208,
        92078.28864, 130903.392384, 136314.3936, 20296.2504192, 13092.8196624,
        184405.536, 13114.5725472, 35131.9909824, 135312.900096, 58760.72928,
        58792.49712, 130821.79104, 0, 31.7678400000004, 0, 49.7265599999955,
        0, 184405.536, 58776.6132, 130821.79104, 92053.42536, 130903.392384,
        184405.536, 58776.6132, 130821.79104, 92053.42536, 130903.392384,
        39756.9975148861, 61399.3898859055, 158611.45156489, 85734.7512321073,
        88173.5230833176, 100180.9091136, 116018.1431808, 88173.5230833176,
        61399.3898859055, 86206.1380539059, 72670.0113017018, 139659.823843051,
        154016.100176007, 68281.3489438994, 99902.8263281649, 119875.425704057,
        89708.4106789686, 119875.425704057, 89708.4106789686, 83531.4346645223,
        85717.9722025068, 70438.6150021223, 72603.3996553068, 48312.1271634612,
        49416.2723748534, 48312.1271634612, 49416.2723748534, 84008.9161793734,
        84000.1163679774, 74611.3351969121, 84008.9161793734, 108873.49470408,
        363044.43512534, 0, 363044.43512534, 363044.43512534
      ),
      temperature = c(
        69.3, 69.4, 68.6, 70, 69.2, 71.3, 70,
        71.4, 69.3, 68.8, 69.2, 69.3, 68.5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 69.3, 71.3, 70, 71.4, 69.3, 69.4, 68.8, 69.2, 69.3,
        68.5, 68.6, 70, 69.2, 69.3, 71.3, 70, 71.4, 69.3, 69.4, 68.8,
        69.2, 69.3, 68.5, 68.6, 70, 69.2, 69.4278239587787, 69.5278911169924,
        68.9409467882664, 70.2839728060707, 69.2528582981929, 71.3545393924532,
        70.256104808548, 71.4546291366507, 69.3914942206321, 69.0819370799837,
        69.2816174964647, 69.3816604058404, 68.8407392065235, 0, 0.100042909375702,
        0, 0.100067158213761, 0, 70.256104808548, 69.3316389511525, 68.8407392065235,
        69.4778575378855, 68.9409467882664, 70.256104808548, 69.3316389511525,
        68.8407392065235, 69.4778575378855, 68.9409467882664, 69.5054672527698,
        69.100858920744, 70.4763780811205, 69.3911797478258, 69.0703868293646,
        0.321107391267475, 1.18311388532665, 69.0819370799837, 69.2528582981929,
        69.1812727099935, 69.5458966750026, 69.3647723898868, 69.2053583418366,
        0.185821405989287, 0.140694862882995, 69.2982690448312, 69.4351198213283,
        69.2982690448312, 69.4351198213283, 69.4661966343991, 69.3301599482243,
        1.88834275805405, 2.12446918842645, 70.4103680134262, 70.3923945424375,
        70.4103680134262, 70.3923945424375, 70.4390564384179, 70.4210800266803,
        0.0552980544401862, 70.4390564384179, 70.4455048487396, 70.4942576977863,
        0, 70.4942576977863, 70.4942576977863),
      pressure = c(
        0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399,
        0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0.588399, 0.588399, 0.588399, 0.588399,
        0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399,
        0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399,
        0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399, 0.588399,
        0.588399, 0.589553153147965, 0.589553205630761, 0.603148919592741,
        0.6002790447058, 0.594141977424899, 0.605537025916663, 0.590703699097637,
        0.605537834921229, 0.598322453871811, 0.595479756020135, 0.589136297906226,
        0.58913633136859, 0.603148259317172, 0, 3.34623638797993e-08,
        0, 5.24827954428986e-08, 0, 0.590703699097637, 0.589136314637408,
        0.603148259317172, 0.589553179389363, 0.603148919592741, 0.590703699097637,
        0.589136314637408, 0.603148259317172, 0.589553179389363, 0.603148919592741,
        0.589944514371461, 0.611972974956264, 0.592683526770374, 0.589980879169265,
        0.617868266212428, 0.0223885101922928, 0.0178309975313652, 0.598322453871811,
        0.6002790447058, 0.603890158701458, 0.602131332362321, 0.608400743374275,
        0.604109457151013, 0.014128577981748, 0.0184562290028143, 0.597045168160139,
        0.599172628872868, 0.597045168160139, 0.599172628872868, 0.602030080371704,
        0.599979559057091, 0.00350694554495845, 0.00555827586413815,
        0.603783553144184, 0.60275869698916, 0.603783553144184, 0.60275869698916,
        0.607174909398726, 0.606150021573465, 0.0144913826283515, 0.606150021573465,
        0.602002819247522, 0.613560201407145, 0, 0.613560201407145, 0.613560201407145
      ),
      flow_rate = c(
        30, 30, 16, 20, 16, 10, 30, 10, 16, 20, 30,
        30, 16, 30, 10, 30, 10, 16, 30, 20, 30, 30, 16, 16, 20, 16, 30,
        10, 30, 10, 16, 30, 20, 30, 30, 16, 16, 20, 16, 30, 10, 30, 10,
        16, 30, 20, 30, 30, 16, 16, 20, 16, 30, 30, 16, 20, 16, 10, 30,
        10, 16, 20, 30, 30, 16, 30, 60, 16, 60, 16, 30, 60, 16, 60, 16,
        30, 60, 16, 60, 16, 60, 16, 30, 60, 16, 52, 52, 52, 52, 52, 52,
        52, 52, 112, 112, 112, 112, 112, 112, 112, 112, 122, 122, 122,
        122, 122, 122, 122, 122, 274, 274, 274, 274, 274, 274, 274
      ),
      job = unlist(mapply(\(x, y) rep(x, each = y), seq(0, 6, 1), c(52, 28, 11,  8,  8,  5,  4) ))
    ),
    row.names = c(
      "1", "2", "3", "7", "9", "10", "15", "17", "19", "21", "24", "25",
      "26", "span.1", "span.10", "span.15", "span.17", "span.19", "span.2",
      "span.21", "span.24", "span.25", "span.26", "span.3", "span.7",
      "span.9", "median.1", "median.10", "median.15", "median.17",
      "median.19", "median.2", "median.21", "median.24", "median.25",
      "median.26", "median.3", "median.7", "median.9", "mean.1", "mean.10",
      "mean.15", "mean.17", "mean.19", "mean.2", "mean.21", "mean.24",
      "mean.25", "mean.26", "mean.3", "mean.7", "mean.9", "14", "22",
      "31", "4", "5", "6", "71", "8", "91", "101", "11", "12", "13",
      "span.14", "span.22", "span.23", "span.4", "span.5", "median.14",
      "median.22", "median.23", "median.4", "median.5", "mean.14",
      "mean.22", "mean.23", "mean.4", "mean.5", "16", "23", "32", "41",
      "51", "span.20", "span.6", "median.20", "median.6", "mean.20",
      "mean.6", "18", "27", "span.18", "span.8", "median.18", "median.8",
      "mean.18", "mean.8", "110", "28", "span.11", "span.16", "median.11",
      "median.16", "mean.11", "mean.16", "111", "29", "span", "median",
      "mean", "112", "span1", "median1", "mean1"
    ),
    class = "data.frame"
  )
  rm(aggregation_method)


  expect_equal(all(colnames(m325tracebw_report) == colnames(ensample_1)), TRUE)

  expect_equal(all(m325tracebw_report[["node"]]        == ensample_1[["node"]])       , TRUE)
  expect_equal(all(m325tracebw_report[["tracing"]]     == ensample_1[["tracing"]])    , TRUE)
  expect_equal(all(m325tracebw_report[["backward"]]    == ensample_1[["backward"]])   , TRUE)
  expect_equal(all(m325tracebw_report[["aggregation"]] == ensample_1[["aggregation"]]), TRUE)
  expect_equal(all(m325tracebw_report[["job"]]         == ensample_1[["job"]])        , TRUE)

  expect_equal(all.equal(m325tracebw_report[["temperature"]], ensample_1[["temperature"]], tolerance = 1e-8), TRUE)
  expect_equal(all.equal(m325tracebw_report[["pressure"]]   , ensample_1[["pressure"]]   , tolerance = 1e-8), TRUE)
  expect_equal(all.equal(m325tracebw_report[["flow_rate"]]  , ensample_1[["flow_rate"]]  , tolerance = 1e-8), TRUE)

  expect_equal(all.equal(m325tracebw_report[["Q"]]     , ensample_1[["Q"]]   , tolerance = 1e-8), TRUE)
  expect_equal(all.equal(m325tracebw_report[["loss"]]  , ensample_1[["loss"]], tolerance = 1e-8), TRUE)
  expect_equal(all.equal(m325tracebw_report[["flux"]]  , ensample_1[["flux"]], tolerance = 1e-8), TRUE)
})

test_that("*m325tracefw* and *m325tracebw* have the same colnames", {
  expect_equal(
    all(colnames(m325tracefw_report) == colnames(m325tracebw_report)),
    TRUE
  )
})
