


################################################################################


#' Repair and check Rd files
#'
#' The \code{repair_docu} methods repair some sometimes encountered, minor
#' problems in Rd files generated by \pkg{roxygen2} (but not those related to S4
#' methods), and run the checks implemented in \code{check_keywords} and
#' \code{check_examples}.
#'
#' @inheritParams pkg_files
#' @param x For \code{repair_docu}, an object of class \sQuote{Rd}, or name(s)
#'   of Rd file(s) or of one to several package directories. See
#'   \code{\link{pkg_files}} for the possible expansion of names.
#'
#'   For \code{check_keywords}, an object of class \sQuote{Rd}, vector, or
#'   \code{NULL}. \sQuote{Rd} objects are checked as described below. If
#'   \code{NULL}, the total list of known keywords is returned as character
#'   vector, otherwise only those contained in \code{x}. If a logical vector,
#'   \code{x} is used like the \sQuote{full} argument. If a character vector,
#'   used to select the descriptions of certain keywords.
#'
#'   For \code{check_examples}, an object of class \sQuote{Rd}.
#' @param remove.dups Logical scalar indicating whether (probably) duplicate
#'   links in the \sQuote{seealso} section shall be removed.
#' @param file Optional character scalar indicating the filename from which the
#'   \sQuote{Rd} object was read. Useful when creating messages describing a
#'   problem. See \code{\link{logfile}}.
#' @param full Logical scalar indicating whether the full content of the
#'   \sQuote{KEYWORDS.db} file should be returned, not just the list of
#'   keywords.
#' @param list.only Logical scalar indicating whether the keywords should just
#'   be collected, not checked against the database.
#'
#' @export
#' @return The \sQuote{Rd} method of \code{repair_docu} returns an object of
#'   class \sQuote{Rd}, the character method a list of such objects, invisibly.
#'
#'   \code{check_keywords} returns a character vector. As a side effect, problem
#'   messages are printed to \code{stderr}. See \code{\link{logfile}} for how to
#'   send these messages to a file.
#'
#'   \code{check_examples} returns a logical scalar, \code{TRUE} if examples are
#'   either there and should be there or are missing and should be missing,
#'   \code{FALSE} otherwise. As a side effect, problem messages are printed to
#'   \code{stderr}. See \code{\link{logfile}} for how to send these messages to
#'   a file.
#'
#' @details
#'   \code{check_keywords} checks keywords in Rd files for their presence, for
#'   their occurrence within the list given in the \sQuote{KEYWORDS.db} file
#'   that comes with \R{}, and for fitting to the \sQuote{docType} of the
#'   \sQuote{Rd} object. The \sQuote{Rd} method examines the content of Rd
#'   files.
#'
#'   In addition to checking that the keywords used are listed in the
#'   \sQuote{KEYWORDS.db} file, it is checked that \sQuote{internal} is not
#'   combined with other keywords, and that \sQuote{docType} entries such as
#'   \sQuote{class}, \sQuote{data} or \sQuote{package} are accompanied by the
#'   appropriate keywords.
#'
#'   \code{check_examples} does \strong{not} run the examples, if any, and check
#'   their results. It checks presence/absence of examples in Rd files. They
#'   should not be present if the list of \sQuote{keywords} contains
#'   \sQuote{internal}. Otherwise they should be present unless \sQuote{docType}
#'   is \sQuote{class} or \sQuote{package}.
#'
#' @family rd-functions
#' @keywords package
#' @examples
#'
#' # 'Rd' objects
#' summary(x <- repair_docu(xyplot.docu))
#' stopifnot(identical(x, xyplot.docu)) # nothing to repair here
#' (x <- check_keywords(xyplot.docu))
#' stopifnot(identical(x, "hplot"))
#' (x <- check_examples(xyplot.docu))
#' stopifnot(x)
#'
#' # Character data, or zero arguments
#' (x <- check_keywords())
#' stopifnot(length(x) > 1, is.character(x), is.null(names(x)))
#' (x <- check_keywords(c("aplot", "loess")))
#' stopifnot(length(x) == 2, is.character(x), names(x) == c("aplot", "loess"))
#'
repair_docu <- function(x, ...) UseMethod("repair_docu")

#' @rdname repair_docu
#' @method repair_docu character
#' @export
#'
repair_docu.character <- function(x, ignore = NULL, ...) {
  do_repair <- function(x) {
    data <- repair_docu(parse_Rd(file = x), ...)
    check_keywords(data, file = x, list.only = FALSE)
    check_examples(data, file = x)
    puts(data, file = x)
  }
  x <- pkg_files(x, what = "man", installed = FALSE, ignore = ignore)
  invisible(sapply(x, do_repair, simplify = FALSE))
}

#' @rdname repair_docu
#' @method repair_docu Rd
#' @export
#'
repair_docu.Rd <- function(x, remove.dups = FALSE, ...) {
  cum_parts <- function(x) {
    x <- strsplit(x, ".", fixed = TRUE)
    x <- x[vapply(x, length, 0L) > 0L]
    unlist(lapply(x, function(y) {
      vapply(seq_along(y),
        function(i) paste0(y[seq.int(1L, i)], collapse = "."), "")
    }))
  }
  function.names <- new.env(parent = emptyenv())
  removed <- FALSE
  do_repair <- function(x, parent.tags) {
    case(attr(x, "Rd_tag"),
      TEXT = {
        switch(parent.tags[1L],
          `\\keyword` = x[x == "dataset"] <- "datasets",
          `\\link` = if (remove.dups && "\\seealso" %in% parent.tags[-1L]) {
            for (part in cum_parts(x))
              if (exists(part, function.names)) {
                x <- NULL
                removed <<- TRUE
                break
              }
            if (!is.null(x)) {
              removed <<- FALSE
              function.names[[x]] <- NULL
            }
          },
          `\\seealso` = if (removed) {
            x <- NULL
            removed <<- FALSE
          }
        )
        x
      },
      COMMENT =,
      VERB = x,
      RCODE = {
        switch(parent.tags[1L],
          `\\usage` = {
            if (grepl("\\s*<-\\s*value\\s*$", x, perl = TRUE))
            # Repair duplicate 'value' entries of replacement functions
              x <- sub(",\\s*value", "", x, perl = TRUE)
            # break long lines
            x <- gsub("(.{80})\\s", "\\1\n    ", x, perl = TRUE)
          }
        )
        x
      }
    )
  }
  repair_recursively <- function(x, parent.tags) {
    if (!is.list(x))
      return(do_repair(x, parent.tags))
    if (!length(x))
      return(x) # keep lists as-is if they were already empty
    if (is.null(this.tag <- attr(x, "Rd_tag")))
      this.tag <- ".empty"
    y <- lapply(x, repair_recursively, parent.tags = c(this.tag, parent.tags))
    y[vapply(y, is.null, NA)] <- NULL
    if (!length(y))
      return(NULL)
    attributes(y) <- attributes(x)
    y
  }
  LL(remove.dups)
  repair_recursively(x, ".toplevel")
}

#' @rdname repair_docu
#' @export
#'
check_keywords <- function(x, ...) UseMethod("check_keywords")

#' @rdname repair_docu
#' @method check_keywords NULL
#' @export
#'
check_keywords.NULL <- function(x, full = FALSE, ...) {
  x <- readLines(file.path(R.home("doc"), "KEYWORDS.db"))
  x <- grep("|", x, fixed = TRUE, value = TRUE)
  x <- sub("^.*\\|", "", x, perl = TRUE)
  x <- do.call(rbind, strsplit(x, "\\s*:\\s*", perl = TRUE))
  if (L(full))
    return(structure(x[, 2L], names = x[, 1L]))
  sort.int(unique.default(x[, 1L]))
}

#' @rdname repair_docu
#' @method check_keywords character
#' @export
#'
check_keywords.character <- function(x, ...) {
  check_keywords(x = NULL, full = TRUE, ...)[x]
}

#' @rdname repair_docu
#' @method check_keywords logical
#' @export
#'
check_keywords.logical <- function(x, ...) {
  check_keywords(x = NULL, full = x, ...)
}

#' @rdname repair_docu
#' @method check_keywords Rd
#' @export
#'
check_keywords.Rd <- function(x, file = NULL, list.only = FALSE, ...) {
  if (!length(kw <- which(subset(x, "keyword")))) {
    if (L(list.only)) {
      problem("no keywords", file)
      return(character())
    } else
      return(invisible(character()))
  }
  kw <- unlist(x[kw])
  if (L(list.only))
    return(kw)
  if (exists("rd.keywords", PKGUTILS_OPTIONS))
    known.kw <- PKGUTILS_OPTIONS$rd.keywords
  else
    PKGUTILS_OPTIONS$rd.keywords <- known.kw <- check_keywords()
  if (length(bad <- setdiff(kw, known.kw)))
    problem(paste("unknown keywords:", paste0(bad, collapse = ", ")), file)
  else if ("internal" %in% kw && length(kw) > 1L)
    problem("'internal' combined with other keyword(s)", file)
  else if (length(doctype <- unlist(subset(x, "docType", values = TRUE))))
    switch(doctype,
      class = if (!all(c("classes", "methods") %in% kw))
        problem("'class' docType but no 'classes' or no 'methods' keyword",
          file),
      data = if (!"datasets" %in% kw)
        problem("'data' docType but no 'datasets' keyword", file),
      package = if (!"package" %in% kw)
        problem("'package' docType but no 'package' keyword", file)
    )
  invisible(kw)
}

#' @rdname repair_docu
#' @export
#'
check_examples <- function(x, ...) UseMethod("check_examples")

#' @rdname repair_docu
#' @method check_examples Rd
#' @export
#'
check_examples.Rd <- function(x, file = NULL, ...) {
  kw <- check_keywords(x = x, file = file, list.only = TRUE, ...)
  if (length(ex <- which(subset(x, "examples"))) && "internal" %in% kw) {
    problem("'internal' keyword but examples present", file)
    FALSE
  } else if (!length(ex) && !"internal" %in% kw) {
    doctype <- unlist(subset(x, "docType", values = TRUE))
    if (all(doctype %in% c("class", "package")))
      TRUE
    else {
      problem("no 'internal' keyword but examples missing", file)
      FALSE
    }
  } else
    TRUE
}


################################################################################


#' Repair S4 documentation in Rd files
#'
#' Repair S4 documentation in Rd files by adding \sQuote{docType},
#' \sQuote{usage}, and \sQuote{alias} entries.
#'
#' @param x Character vector of names of input files, or names of \R package
#'   directories. The latter will be expanded as appropriate. \code{x} is
#'   passed to \code{\link{pkg_files}} with the \sQuote{installed} argument
#'   set to \code{FALSE}. See there for further details.
#' @param ... Optional arguments, currently passed as arguments additional to
#'   \sQuote{x} to \code{\link{run_ruby}}. See there for details.
#' @param ignore \code{NULL} or character vector with names of files to ignore.
#'   Passed to \code{\link{pkg_files}}, see there for details of how names
#'   are matched.
#' @return Currently the return value of the call to \code{\link{run_ruby}}.
#' @details This reparation process is currently implemented in a Ruby script
#'   that comes with the package. It is automatically found in the installation
#'   directory but fails if a suitable version of Ruby, i.e. \eqn{\ge 1.9.0},
#'   is unavailable. See \code{\link{run_ruby}} for further details.
#'
#'   This function is intended for augmenting Rd documentation generated by
#'   \pkg{roxygen2}, as long as it does not fully support S4. In addition
#'   to the generation of missing documentation entries, \code{repair_S4_docu}
#'   can also recognize S4 methods generated within a loop provided a specific
#'   annotation is used. An example is given below.
#'
#'   Another annotation can be used for redirecting the generated entries from
#'   one Rd file to another one. The file extension and the directory path
#'   would be added automatically. See the examples.
#' @export
#' @family rd-functions
#' @keywords package
#' @examples
#' ## this is an example for the redirection of Rd file names:
#' #= dim length
#' ## i.e., text that would be sent to 'dim.Rd' would now instead be written
#' ## to 'length.Rd'
#'
#' ## here is an example for automatically generated S4 methods from the opm
#' ## package:
#' \dontrun{
#'   lapply(c(
#'     #+
#'     wells,
#'     plate_type
#'     #-
#'   ), FUN = function(func_) {
#'     setMethod(func_, OPMS, function(object, ...) {
#'       func_(plates(object)[[1L]], ...)
#'     }, sealed = SEALED)
#'   })
#' }
#' ## method names between '#+' and '#-' were collected and their usage
#' ## generated from the subsequent setMethod() call invoking a 'func_'
#' ## variable.
#'
#' ## See also the 'docu.R' script provided with the pkgutils package, option
#' ## '--s4methods'.
#'
repair_S4_docu <- function(x, ...) UseMethod("repair_S4_docu")

#' @rdname repair_S4_docu
#' @method repair_S4_docu character
#' @export
#'
repair_S4_docu.character <- function(x, ..., ignore = NULL) {
  run_pkgutils_ruby(x = x, script = "repair_S4_docu.rb", ignore = ignore, ...)
}


################################################################################


