#' Wrapper for mixOmics pls functions
#'
#' Based on arguments, this wrapper routes the data and arguments to the four
#' `pls` functions that are sparse/dense or regression/classification.
#' @param x A data frame or matrix of predictors.
#' @param y For classification, a factor. For regression, a matrix, vector, or
#' data frame.
#' @param ncomp The number of PLS components. If left null, the maximum possible
#' is used.
#' @param keepX The number of non-zero loadings _per component_. If the value is
#' a vector, the value is left as-is. Otherwise, the scalar is expanded to be
#' the same for all components. If NULL, either [mixOmics::pls()] or
#' [mixOmics::plsda()] are used. Otherwise, their sparse analogs are used.
#' @return A model object generated by [mixOmics::pls()], [mixOmics::plsda()],
#' [mixOmics::spls()], or [mixOmics::splsda()].
#' @export
#' @keywords internal
pls_fit <- function(x, y, ncomp = NULL, keepX = NULL, ...) {
  p <- ncol(x)
  if (!is.matrix(x)) {
    x <- as.matrix(x)
  }
  if (is.null(ncomp)) {
    ncomp <- p
  } else {
    ncomp <- min(ncomp, p)
  }
  if (!is.null(keepX) && length(keepX) == 1) {
    keepX <- rep(min(keepX, p), p)
  }
  if (is.factor(y)) {
    if (is.null(keepX)) {
      res <- mixOmics::plsda(X = x, Y = y, ncomp = ncomp, ...)
    } else {
      res <- mixOmics::splsda(X = x, Y = y, ncomp = ncomp, keepX = keepX, ...)
    }
  } else {
    if (is.null(keepX)) {
      res <- mixOmics::pls(X = x, Y = y, ncomp = ncomp, ...)
    } else {
      res <- mixOmics::spls(X = x, Y = y, ncomp = ncomp, keepX = keepX, ...)
    }
  }
  res
}
