% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/write_testthat_file.R
\name{write_testthat_file}
\alias{write_testthat_file}
\title{Transform a \strong{pointblank} agent to a \strong{testthat} test file}
\usage{
write_testthat_file(
  agent,
  name = NULL,
  path = NULL,
  overwrite = FALSE,
  skips = NULL,
  quiet = FALSE
)
}
\arguments{
\item{agent}{An agent object of class \code{ptblank_agent}.}

\item{name}{An optional name for for the \strong{testhat} test file. This should
be a name without extension and without the leading \code{"test-"} text. If
nothing is supplied, the name will be derived from the \code{tbl_name} in the
agent. If that's not present, a generic name will be used.}

\item{path}{A path can be specified here if there shouldn't be an attempt to
place the file in \code{testthat/tests}.}

\item{overwrite}{Should a \strong{testthat} file of the same name be overwritten?
By default, this is \code{FALSE}.}

\item{skips}{This is an optional vector of test-skipping keywords modeled
after the \strong{testthat} \verb{skip_on_*()} functions. The following keywords can
be used to include \verb{skip_on_*()} statements: \code{"cran"}
(\code{\link[testthat:skip]{testthat::skip_on_cran()}}), \code{"travis"} (\code{\link[testthat:skip]{testthat::skip_on_travis()}}),
\code{"appveyor"} (\code{\link[testthat:skip]{testthat::skip_on_appveyor()}}), \code{"ci"}
(\code{\link[testthat:skip]{testthat::skip_on_ci()}}), \code{"covr"} (\code{\link[testthat:skip]{testthat::skip_on_covr()}}), \code{"bioc"}
(\code{\link[testthat:skip]{testthat::skip_on_bioc()}}). There are keywords for skipping tests on
certain operating systems and all of them will insert a specific
\code{\link[testthat:skip]{testthat::skip_on_os()}} call. These are \code{"windows"}
(\code{skip_on_os("windows")}), \code{"mac"} (\code{skip_on_os("mac")}), \code{"linux"}
(\code{skip_on_os("linux")}), and \code{"solaris"} (\code{skip_on_os("solaris")}). These
calls will be placed at the top of the generated \strong{testthat} test file.}

\item{quiet}{Should the function \emph{not} inform when the file is written? By
default this is \code{FALSE}.}
}
\value{
Invisibly returns \code{TRUE} if the \strong{testthat} file has been written.
}
\description{
With a \strong{pointblank} \emph{agent}, we can write a \strong{testthat} test file and opt
to place it in the \code{testthat/tests} if it is available in the project path
(we can specify an alternate path as well). This works by transforming the
validation steps to a series of \verb{expect_*()} calls inside individual
\code{\link[testthat:test_that]{testthat::test_that()}} statements.

A hard requirement for using \code{write_testthat_file()} on an agent is the
presence of a \code{read_fn}, which is a function that is invoked to obtain the
target table. The \code{read_fn} statement will be placed at the top of the
\strong{testthat} test file so that the target table is available for each of the
\code{\link[testthat:test_that]{testthat::test_that()}} statements that follow. If an \emph{agent} does not have a
\code{read_fn} it can be added via the \code{\link[=set_read_fn]{set_read_fn()}}.

Thresholds will be obtained from those applied for the \code{stop} state. This can
be set up for a \strong{pointblank} \emph{agent} by passing an \code{action_levels} object
to the \code{actions} argument of \code{\link[=create_agent]{create_agent()}} or the same argument of any
included validation function. If \code{stop} thresholds are not available, then a
threshold value of \code{1} will be used for each generated \verb{expect_*()} statement
in the resulting \strong{testthat} test file.

There is no requirement that the \strong{agent} first undergo interrogation with
\code{\link[=interrogate]{interrogate()}}. However, it may be useful as a dry run to interactively
perform an interrogation on the target data before generating the
\strong{testthat} test file.
}
\details{
Tests for inactive validation steps will be skipped with a clear message
indicating that the reason for skipping was due to the test not being active.
Any inactive validation steps can be forced into an active state by using the
\code{\link[=activate_steps]{activate_steps()}} on an \emph{agent} (the opposite is possible with the
\code{\link[=deactivate_steps]{deactivate_steps()}} function).

The \strong{testthat} package comes with a series of \verb{skip_on_*()} functions which
conveniently cause the test file to be skipped entirely if certain conditions
are met. We can quickly set any number of these at the top of the
\strong{testthat} test file by supplying keywords as a vector to the \code{skips}
option of \code{write_testthat_file()}. For instance, setting
\verb{skips = c("cran", "windows)} will add the \strong{testthat} \code{skip_on_cran()} and
\code{skip_on_os("windows")} statements, meaning that the generated test file
won't run on a CRAN system or if the system OS is Windows.

Here is an example of \strong{testthat} test file output:

\code{test-small_table.R}\preformatted{# Generated by pointblank

tbl <- small_table

test_that("column `date_time` exists", \{
  
  expect_col_exists(
    tbl,
    columns = vars(date_time),
    threshold = 1
  ) 
\})

test_that("values in `c` should be <= `5`", \{
  
  expect_col_vals_lte(
    tbl,
    columns = vars(c),
    value = 5,
    threshold = 0.25
  ) 
\})

}

This was generated by the following set of statements:\preformatted{library(pointblank)

agent <- 
  create_agent(
    read_fn = ~ small_table,
    actions = action_levels(stop_at = 0.25)
  ) \%>\%
  col_exists(vars(date_time)) \%>\%
  col_vals_lte(vars(c), value = 5)
  
write_testthat_file(
  agent = agent,
  name = "small_table",
  path = "."
)
}
}
\section{Function ID}{

8-5
}

\examples{
if (interactive()) {

# Creating an `action_levels` object is a
# common workflow step when creating a
# pointblank agent; we designate failure
# thresholds to the `warn`, `stop`, and
# `notify` states using `action_levels()`
al <- 
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )

# A pointblank `agent` object is now
# created and the `al` object is provided
# to the agent; the static thresholds
# provided by `al` make reports a bit
# more useful after interrogation
agent <- 
  create_agent(
    read_fn = ~ small_table,
    label = "An example.",
    actions = al
  ) \%>\%
  col_exists(vars(date, date_time)) \%>\%
  col_vals_regex(
    vars(b),
    regex = "[0-9]-[a-z]{3}-[0-9]{3}"
  ) \%>\%
  col_vals_gt(vars(d), value = 100) \%>\%
  col_vals_lte(vars(c), value = 5) \%>\%
  interrogate()

# This agent and all of the checks can
# be transformed into a testthat file
# with `write_testthat_file()`; the `stop`
# thresholds will be ported over
write_testthat_file(
  agent = agent,
  name = "small_table",
  path = "."
)

# The above code will generate a file with
# the name `test-small_table.R`; the path
# was specified with `"."` but, by default,
# the function will place the file in the
# `tests/testthat` folder if it's available

# An agent on disk as a YAML file can be
# made into a testthat file; the
# 'agent-small_table.yml' file is
# available in the package through
# `system.file()`
yml_file <- 
  system.file(
    "yaml", "agent-small_table.yml",
    package = "pointblank"
  )

# Writing the testthat file into the
# working directory is much the same
# as before but we're reading the
# agent from disk this time
write_testthat_file(
  agent = yaml_read_agent(yml_file),
  name = "from_agent_yaml",
  path = "."
)

}

}
\seealso{
Other Post-interrogation: 
\code{\link{all_passed}()},
\code{\link{get_agent_x_list}()},
\code{\link{get_data_extracts}()},
\code{\link{get_sundered_data}()}
}
\concept{Post-interrogation}
