% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bitwise.r
\name{win.ia}
\alias{win.ia}
\title{Calculate windows of the index of association for genlight objects.}
\usage{
win.ia(x, window = 100L, min.snps = 3L, threads = 1L, quiet = FALSE,
  name_window = TRUE, chromosome_buffer = TRUE)
}
\arguments{
\item{x}{a \link[=genlight-class]{genlight} or \link[=snpclone-class]{snpclone} object.}

\item{window}{an integer specifying the size of the window.}

\item{min.snps}{an integer specifying the minimum number of snps allowed per
window. If a window does not meet this criteria, the value will return as
\code{NA}.}

\item{threads}{The maximum number of parallel threads to be used within this
function. Defaults to 1 thread, in which the function will run serially. A
value of 0 will attempt to use as many threads as there are available
cores/CPUs. In most cases this is ideal for speed. Note: this option is
passed to \code{\link[=bitwise.ia]{bitwise.ia()}} and does not parallelize the windowization process.}

\item{quiet}{if \code{FALSE} (default), a progress bar will be printed to the screen.}

\item{name_window}{if \code{TRUE} (default), the result vector will be named with
the terminal position of the window. In the case where several chromosomes
are represented, the position will be appended using a period/full stop.}

\item{chromosome_buffer}{\emph{DEPRECATED} if \code{TRUE} (default), buffers will be placed
between adjacent chromosomal positions to prevent windows from spanning two
chromosomes.}
}
\value{
A value of the standardized index of association for all windows in
each chromosome.
}
\description{
Genlight objects can contain millions of loci. Since it does not make much
sense to calculate the index of association over that many loci, this
function will scan windows across the loci positions and calculate the index
of association.
}
\note{
this will calculate the standardized index of association from Agapow
and Burt, 2001. See \code{\link[=ia]{ia()}} for details.
}
\examples{

# with structured snps assuming 1e4 positions
set.seed(999)
x <- glSim(n.ind = 10, n.snp.nonstruc = 5e2, n.snp.struc = 5e2, ploidy = 2)
position(x) <- sort(sample(1e4, 1e3))
res <- win.ia(x, window = 300L) # Calculate for windows of size 300
plot(res, type = "l")

\dontrun{

# unstructured snps
set.seed(999)
x <- glSim(n.ind = 10, n.snp.nonstruc = 1e3, ploidy = 2)
position(x) <- sort(sample(1e4, 1e3))
res <- win.ia(x, window = 300L) # Calculate for windows of size 300
plot(res, type = "l")

# Accounting for chromosome coordinates
set.seed(999)
x <- glSim(n.ind = 10, n.snp.nonstruc = 5e2, n.snp.struc = 5e2, ploidy = 2)
position(x) <- as.vector(vapply(1:10, function(x) sort(sample(1e3, 100)), integer(100)))
chromosome(x) <- rep(1:10, each = 100)
res <- win.ia(x, window = 100L)
plot(res, type = "l")

# Converting chromosomal coordinates to tidy data
library("dplyr")
library("tidyr")
res_tidy <- res \%>\% 
  data_frame(rd = ., chromosome = names(.)) \%>\% # create two column data frame
  separate(chromosome, into = c("chromosome", "position")) \%>\% # get the position info
  mutate(position = as.integer(position)) \%>\% # force position as integers
  mutate(chromosome = factor(chromosome, unique(chromosome))) # force order chromosomes
res_tidy

# Plotting with ggplot2
library("ggplot2")
ggplot(res_tidy, aes(x = position, y = rd, color = chromosome)) +
  geom_line() +
  facet_wrap(~chromosome, nrow = 1) +
  ylab(expression(bar(r)[d])) +
  xlab("terminal position of sliding window") +
  labs(caption = "window size: 100bp") + 
  theme(axis.text.x = element_text(angle = 90, hjust = 1, vjust = 0.5)) +
  theme(legend.position = "top")

}

}
\seealso{
\link[=genlight-class]{genlight}, \link[=snpclone-class]{snpclone}, \code{\link[=ia]{ia()}}, \code{\link[=samp.ia]{samp.ia()}}, \code{\link[=bitwise.dist]{bitwise.dist()}}
}
\author{
Zhian N. Kamvar, Jonah C. Brooks
}
