% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/etc_utils_fortify.R, R/g_fortify.R
\name{fortify}
\alias{fortify}
\alias{fortify.mmcurves}
\alias{fortify.mmpoints}
\alias{fortify.mscurves}
\alias{fortify.mspoints}
\alias{fortify.smcurves}
\alias{fortify.smpoints}
\alias{fortify.sscurves}
\alias{fortify.sspoints}
\title{Convert a curves and points object to a data frame for ggplot2}
\usage{
\method{fortify}{sscurves}(model, raw_curves = TRUE, ...)

\method{fortify}{mscurves}(model, raw_curves = TRUE, ...)

\method{fortify}{smcurves}(model, raw_curves = FALSE, ...)

\method{fortify}{mmcurves}(model, raw_curves = FALSE, ...)

\method{fortify}{sspoints}(model, raw_curves = TRUE, ...)

\method{fortify}{mspoints}(model, raw_curves = TRUE, ...)

\method{fortify}{smpoints}(model, raw_curves = FALSE, ...)

\method{fortify}{mmpoints}(model, raw_curves = FALSE, ...)
}
\arguments{
\item{model}{An \code{S3} object generated by \code{\link{evalmod}}.
  The \code{fortify} function takes one of the following \code{S3} objects.

\enumerate{

  \item ROC and Precision-Recall curves (mode = "rocprc")

  \tabular{lll}{
    \strong{\code{S3} object}
    \tab \strong{# of models}
    \tab \strong{# of test datasets} \cr

    sscurves \tab single   \tab single   \cr
    mscurves \tab multiple \tab single   \cr
    smcurves \tab single   \tab multiple \cr
    mmcurves \tab multiple \tab multiple
  }

  \item Basic evaluation measures (mode = "basic")

  \tabular{lll}{
    \strong{\code{S3} object}
    \tab \strong{# of models}
    \tab \strong{# of test datasets} \cr

    sspoints \tab single   \tab single   \cr
    mspoints \tab multiple \tab single   \cr
    smpoints \tab single   \tab multiple \cr
    mmpoints \tab multiple \tab multiple
  }
}

See the \strong{Value} section of \code{\link{evalmod}} for more details.}

\item{raw_curves}{A Boolean value to specify whether raw curves are
shown instead of the average curve. It is effective only
when \code{raw_curves} is set to \code{TRUE}
of the \code{\link{evalmod}} function.}

\item{...}{Not used by this method.}
}
\value{
The \code{fortify} function returns a data frame for
  \pkg{ggplot2}.
}
\description{
The \code{fortify} function converts an \code{S3} object generated by
  \code{\link{evalmod}} to a data frame for \pkg{ggplot2}.
}
\examples{

## Load ggplot2
library(ggplot2)

#############################################################################
### Single model & single test dataset
###

## Load a dataset with 10 positives and 10 negatives
data(P10N10)

## Generate an sscurve object that contains ROC and Precision-Recall curves
sscurves <- evalmod(scores = P10N10$scores, labels = P10N10$labels)

## Fortify sscurves
ssdf <- fortify(sscurves)

## Plot a ROC curve
p_roc <- ggplot(subset(ssdf, curvetype == "ROC"), aes(x = x, y = y))
p_roc <- p_roc + geom_line()
\dontrun{p_roc}

## Plot a Precision-Recall curve
p_prc <- ggplot(subset(ssdf, curvetype == "PRC"), aes(x = x, y = y))
p_prc <- p_prc + geom_line()
\dontrun{p_prc}

## Generate an sspoints object that contains basic evaluation measures
sspoints <- evalmod(mode = "basic", scores = P10N10$scores,
                    labels = P10N10$labels)
## Fortify sspoints
ssdf <- fortify(sscurves)

## Plot threshold vs. precision
p_prec <- ggplot(subset(ssdf, curvetype == "precision"), aes(x = x, y = y))
p_prec <- p_prc + geom_point()
\dontrun{p_prec}


#############################################################################
### Multiple models & single test dataset
###

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(1, 100, 100, "all")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]])

## Generate an mscurve object that contains ROC and Precision-Recall curves
mscurves <- evalmod(mdat)

## Fortify mscurves
msdf <- fortify(mscurves)

## Plot ROC curve
df_roc <- subset(msdf, curvetype == "ROC")
p_roc <- ggplot(df_roc, aes(x = x, y = y, color = modname))
p_roc <- p_roc + geom_line()
\dontrun{p_roc}

## Fortified data frame can be used for plotting a Precision-Recall curve
df_prc <- subset(msdf, curvetype == "PRC")
p_prc <- ggplot(df_prc, aes(x = x, y = y, color = modname))
p_prc <- p_prc + geom_line()
\dontrun{p_prc}

## Generate an mspoints object that contains basic evaluation measures
mspoints <- evalmod(mdat, mode = "basic")

## Fortify mspoints
msdf <- fortify(mscurves)

## Plot threshold vs. precision
df_prec <- subset(msdf, curvetype == "precision")
p_prec <- ggplot(df_prec, aes(x = x, y = y, color = modname))
p_prec <- p_prc + geom_point()
\dontrun{p_prec}


#############################################################################
### Single model & multiple test datasets
###

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(10, 100, 100, "good_er")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]],
               dsids = samps[["dsids"]])

## Generate an smcurve object that contains ROC and Precision-Recall curves
smcurves <- evalmod(mdat, raw_curves = TRUE)

## Fortify smcurves
smdf <- fortify(smcurves)

## Plot average ROC curve
df_roc <- subset(smdf, curvetype == "ROC")
p_roc <- ggplot(df_roc, aes(x = x, y = y, ymin = ymin, ymax = ymax))
p_roc <- p_roc + geom_smooth(stat = "identity")
\dontrun{p_roc}

## Plot average Precision-Recall curve
df_prc <- subset(smdf, curvetype == "PRC")
p_prc <- ggplot(df_prc, aes(x = x, y = y, ymin = ymin, ymax = ymax))
p_prc <- p_prc + geom_smooth(stat = "identity")
\dontrun{p_prc}

## Generate an smpoints object that contains basic evaluation measures
smpoints <- evalmod(mdat, mode = "basic")

## Fortify smpoints
smdf <- fortify(smpoints)

## Plot threshold vs. precision
df_prec <- subset(smdf, curvetype == "precision")
p_prec <- ggplot(df_prec, aes(x = x, y = y, ymin = ymin, ymax = ymax))
p_prec <- p_prec + geom_ribbon(aes(min = ymin, ymax = ymax),
                               stat = "identity", alpha = 0.25,
                               fill = "grey25")
p_prec <- p_prec + geom_point(aes(x = x, y = y))
\dontrun{p_prec}


#############################################################################
### Multiple models & multiple test datasets
###

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(10, 100, 100, "all")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]],
               dsids = samps[["dsids"]])

## Generate an mscurve object that contains ROC and Precision-Recall curves
mmcurves <- evalmod(mdat, raw_curves = TRUE)

## Fortify mmcurves
mmdf <- fortify(mmcurves)

## Plot average ROC curve
df_roc <- subset(mmdf, curvetype == "ROC")
p_roc <- ggplot(df_roc, aes(x = x, y = y, ymin = ymin, ymax = ymax))
p_roc <- p_roc + geom_smooth(aes(color = modname), stat = "identity")
\dontrun{p_roc}

## Plot average Precision-Recall curve
df_prc <- subset(mmdf, curvetype == "PRC")
p_prc <- ggplot(df_prc, aes(x = x, y = y, ymin = ymin, ymax = ymax))
p_prc <- p_prc + geom_smooth(aes(color = modname), stat = "identity")
\dontrun{p_prc}

## Generate an mmpoints object that contains basic evaluation measures
mmpoints <- evalmod(mdat, mode = "basic")

## Fortify mmpoints
mmdf <- fortify(mmpoints)

## Plot threshold vs. precision
df_prec <- subset(smdf, curvetype == "precision")
p_prec <- ggplot(df_prec, aes(x = x, y = y, ymin = ymin, ymax = ymax))
p_prec <- p_prec + geom_ribbon(aes(min = ymin, ymax = ymax),
                               stat = "identity", alpha = 0.25,
                               fill = "grey25")
p_prec <- p_prec + geom_point(aes(x = x, y = y, color = modname))
\dontrun{p_prec}

}
\seealso{
\code{\link{evalmod}} for generating \code{S3} objects with
  performance evaluation measures.
  \code{\link{autoplot}} for plotting with \pkg{ggplot2}.
}

